# sd.tcl---Spectrum Display and Analysis application

#    new version July 1997 using common libraries
#     but largely copied from previous versionset-selections

# 6 July 2009 SCL
# this has a bugfix in colourmap handling.
# Each time we want to apply a colourmap we have to
#    rebuild the colourmaps to overcome the behaviour (bug?) of tcl8.5/8.6
#    the 1st element might have become a list itself ( {{#01} #12 #23 #...} )

package require ServerAccess 2.0
package require SpectrumClient

set debug 0
set started 0

set frame SD
set frametitle SD
set icontitle "SpecView"
set helpname sd

set PreFix .sd.s

set preferences(spectrum.DisableTopLevel) 0
set preferences(spectrum.DisableReSize) 0
set DisableTopLevel $preferences(spectrum.DisableTopLevel)    ;# needs to be 1 (true) for some Linux X servers
set DisableReSize $preferences(spectrum.DisableReSize)

set frames {}

set preferences(spectrum.spectrum_height) 450
set preferences(spectrum.cuts_height) 300

set spectrum_height $preferences(spectrum.spectrum_height)
set geometry ""
set cuts_height $preferences(spectrum.cuts_height)

set movement 0

set onedviewopts(scale) 0
set onedviewopts(xmin) 0
set onedviewopts(xmax) 8192
set onedviewopts(ymin) 0
set onedviewopts(ymax) [expr 1024*1024]

set twodviewopts(xmin) 0
set twodviewopts(xmax) 512
set twodviewopts(ymin) 0
set twodviewopts(ymax) 512

set linestyleoptions(spectrum) 1
set linestyleoptions(fit) 2

set tick 1
set mode 0

set showmode "new"
set viewmode "current"
set displaymode "linear"
set slicingmode "off"

set fittags {}

set refreshafter ""
set refreshperiod 10000

set b2tag ""
set b2vertex ""

set colourmap ""
set overlapcolourmap ""

set gate1 ""
set gate2 ""

#    for retain and revert command
set viewring {}
set viewnext 0


set viewmenu  {expand {"zoom in"} {"zoom out"} times2 divide2 left right up down reset log \
  linear {"skip channel 0"} {"scale to view"} undo redo retain revert {refresh Once Off 2 5 10 30 1 0.5 0.25 0.1} \
  {preferences {"Save preferences"} {"view"} {"1d colours"} {"2d colours"} {"2d contours"} Advanced}}

set arrangemenu {{"remove all spectra"} {"remove current spectrum"} {"remove all overlaps"} {"remove last overlap"} rearrange {"example 1d"} {"example 2d"} {"gallery hardcopy"} {"gallery save"}}

set analysismenu {integrate peakfit {"peakfit options..."} brumfit {"brumfit options..."} calibrate... \
  peak/total {"max channel counts"} {"max counts (range)"} {"total counts"} {"counts (range)"} {"Co60 peak/total"} \
  {auto-calibration {"Co60 auto-calibration"} {"Eu152 auto-calibration"} {"auto-calibration options..."}} \
  peakfind {"peakfind options..."} \
  {gainmatching "gainmatch" {"gainmatch options..."} {"gainmatch setup"} {"create NEO++ calibration files"} {"create EB calibration files"}} plot {"plot options..."} \
  {"save cuts"} {"check counts"} {"fix counts"}}

set tagsfitsmenu {{"clear tags"} {"restore tags"} {"clear gates"} {"clear gate"} {"clear pointers"} \
   {"clear pointer"} {"clear fit"} {"polygon editor" {"load polygon array"}  {"save polygon array"} \
    {"select next polygon"} {"create polygon from tags"} {"delete selected polygon"} {"delete all polygons"} \
    {"add vertex"} {"delete vertex"} {"project spectrum"}}}

set showmodesmenu {new add overlap}
set viewmodesmenu {current all}
set displaymodesmenu {linear log}
set slicingmenu   {"slicing off" "slicing on"}


proc mksd {spectrumheight} {
   global PreFix DisableTopLevel
   global env frame gallery preferences
   global viewmenu arrangemenu analysismenu tagsfitsmenu
   global showmodesmenu
   global viewmodesmenu displaymodesmenu
   global slicingmenu
   global spectrum_height heights

   global $frame

   :set footer_position center

   set f [self frame-name]
   set p S[string range $frame [string length $PreFix] end]

if {$DisableTopLevel} then {

      :frame $f  :label "[self frame-title]" :show-footer T :toplevel F  :win-ctrl F \
        :icon-image "@$env(ICONHOME)/spectrum"  \
        :icon-name [self icon-title]

} else {

      :frame $f  :label "[self frame-title]" :show-footer T :toplevel T  \
        :icon-image "@$env(ICONHOME)/spectrum"  \
        :icon-name [self icon-title]

    wm protocol $f  WM_DELETE_WINDOW  "click-quit [self frame-name]"
}


   :panel BUTTONS1 :layout V :padx 5
   :layout H; :col-gap 5; :row-gap 5
   :menubutton [set p]VIEW     :label View          :menu $viewmenu      :helpitem VIEW
   :menubutton [set p]ARRANGE  :label Arrange       :menu $arrangemenu   :helpitem ARRANGE
   :menubutton [set p]ANALYSIS :label Analysis      :menu $analysismenu  :helpitem ANALYSIS
   :menubutton [set p]TAGSFITS :label "Tags & Fits" :menu $tagsfitsmenu  :helpitem TAGSFITS

   :text       [set p]CHANNEL  :label Channel:  :w 12    :helpitem CHANNEL
   :button     [set p]HELP     :label Help :command click-help
   if {[set ${frame}(framemode)] == "spectrum"} then {:button [set p]CLONE :label Clone :command click-another}
   :checkbox   [set p]TICK     :label ""      :helpitem TICK

   :panel BUTTONS2 :layout V :padx 5 :pady 5
   :layout H; :col-gap 5; :row-gap 5
   :button     [set p]RESET    :label Reset   :helpitem RESET
   :button     [set p]REFRESH  :label Refresh :helpitem REFRESH
   :button     [set p]ZOOMIN   :label "@$env(ICONHOME)/expand"       :helpitem ZOOMIN
   :button     [set p]ZOOMOUT  :label "@$env(ICONHOME)/contract"     :helpitem ZOOMOUT
   :button     [set p]TIMES2   :label "@$env(ICONHOME)/times2"       :helpitem TIMES2
   :button     [set p]DIVIDE2  :label "@$env(ICONHOME)/divide2"      :helpitem DIVIDE2
   :button     [set p]LEFT     :label "@$env(ICONHOME)/leftarrow"    :helpitem LEFT
   :button     [set p]RIGHT    :label "@$env(ICONHOME)/rightarrow"   :helpitem RIGHT
   :button     [set p]UP       :label "@$env(ICONHOME)/uparrow"      :helpitem UP
   :button     [set p]DOWN     :label "@$env(ICONHOME)/downarrow"    :helpitem DOWN
   :button     [set p]INVERT   :label "@$env(ICONHOME)/square"       :helpitem INVERT

#    bindings for repetitive button actions

   bind [:path [set p]LEFT]     <ButtonPress-3>   {click-shift-Press %w L}
   bind [:path [set p]LEFT]     <ButtonRelease-3> {click-shift-Release}
   bind [:path [set p]RIGHT]    <ButtonPress-3>   {click-shift-Press %w R}
   bind [:path [set p]RIGHT]    <ButtonRelease-3> {click-shift-Release}
   bind [:path [set p]TIMES2]   <ButtonPress-3>   {click-scale-Press %w U}
   bind [:path [set p]TIMES2]   <ButtonRelease-3> {click-scale-Release}
   bind [:path [set p]DIVIDE2]  <ButtonPress-3>   {click-scale-Press %w D}
   bind [:path [set p]DIVIDE2]  <ButtonRelease-3> {click-scale-Release}
   bind [:path [set p]ZOOMIN]   <ButtonPress-3>   {click-zoom-Press %w I}
   bind [:path [set p]ZOOMIN]   <ButtonRelease-3> {click-zoom-Release}
   bind [:path [set p]ZOOMOUT]  <ButtonPress-3>   {click-zoom-Press %w O}
   bind [:path [set p]ZOOMOUT]  <ButtonRelease-3> {click-zoom-Release}
   bind [:path [set p]UP]       <ButtonPress-3>   {click-view-Press %w U}
   bind [:path [set p]UP]       <ButtonRelease-3> {click-view-Release}
   bind [:path [set p]DOWN]     <ButtonPress-3>   {click-view-Press %w D}
   bind [:path [set p]DOWN]     <ButtonRelease-3> {click-view-Release}

   set mode [set ${frame}(framemode)]
   if {$mode == "spectrum"} then {
      :optionbutton [set p]SHOWMODE      :menu $showmodesmenu      :helpitem SHOWMODE     :w 6
      :optionbutton [set p]VIEWMODE      :menu $viewmodesmenu      :helpitem VIEWMODE     :w 6
      :optionbutton [set p]DISPLAYMODE   :menu $displaymodesmenu   :helpitem DISPLAYMODE  :w 6
      :optionbutton [set p]SLICEMODE     :menu $slicingmenu        :helpitem SLICEMODE
   }

   :panel [set p]SPECTRUM :layout V :h $spectrumheight
   :spectrum $gallery spectrum create   :helpitem SPECTRUM

   :spectrum $gallery configure -twodmode $preferences(spectrum.twodmode)

#    spectrum gallery bindings

   set p [:path $gallery]
   bind $p <Double-ButtonPress-1>          {click-expand}
   bind $p <Shift-Double-ButtonPress-1>    {:spectrum $gallery tag pop 1; click-reset}
   bind $p <ButtonPress-1>                 {click-B1 %W %x %y}
   bind $p <Meta-ButtonPress-1>            {decode-hit %x %y}
   bind $p <Meta-Button1-Motion>           {set-footer "x= %x; y= %y"}
   bind $p <ButtonPress-2>                 {click-B2 %W %x %y}
   bind $p <Button2-Motion>                {click-B2-motion  %x %y}
   bind $p <ButtonRelease-2>               {click-B2-release %x %y}
   bind $p <Double-ButtonPress-2>          {double2 %x %y}
   bind $p <Shift-Double-ButtonPress-2>    {shiftdouble2 %x %y}
   bind $p <ButtonPress-3>                 {click-B3-Press %W %x %y}
   bind $p <ButtonRelease-3>               {click-B3-Release %W %x %y}

#    additional for 2 button mouse
   bind $p <Control-ButtonPress-1>                 {click-B2 %W %x %y}
   bind $p <Control-Button1-Motion>                {click-B2-motion %x %y}
   bind $p <Control-ButtonRelease-1>               {click-B2-release %x %y}
   bind $p <Control-Double-ButtonPress-1>          {double2 %x %y}
   bind $p <Control-Shift-Double-ButtonPress-1>    {shiftdouble2 %x %y}
   bind $p <Control-ButtonPress-3>                 {click-B2 %W %x %y}
   bind $p <Control-Button3-Motion>                {click-B2-motion  %x %y}
   bind $p <Control-ButtonRelease-3>               {click-B2-release %x %y}
   bind $p <Control-Double-ButtonPress-3>          {double2 %x %y}
   bind $p <Control-Shift-Double-ButtonPress-3>    {shiftdouble2 %x %y}

#    keyboard
   set p [:path [self frame-name]]
   bind $p <KeyPress-Up>    {click-updown U}
   bind $p <KeyPress-Down>  {click-updown D}
   bind $p <KeyPress-Left>  {click-shift L}
   bind $p <KeyPress-Right> {click-shift R}
   bind $p <Shift-KeyPress-Up>    {send sb click-show-step -1}
   bind $p <Shift-KeyPress-Down>  {send sb click-show-step 1}

   centre-frame $f
   open-frame $f

   update
   set wh [winfo height $f]
   set heights($mode) "$wh $spectrumheight"
   set ${frame}(geometry) $wh

}

proc viewopts {} {
   global env frame gallery preferences
   :frame [self frame-name].viewopts  :label "View Preferences" :show-footer F  \
        :icon-image "@$env(ICONHOME)/spectrum"  \
        :icon-name "ViewOpts"

   wm protocol [self frame-name].viewopts  WM_DELETE_WINDOW  "click-quit [self frame-name].viewopts"

   :panel CUSTOMVIEW :padx 5 :pady 5 :layout V
   :layout V; :col-gap 5; :row-gap 5
   :message M1 :label "Constants" :bold T
   :slider  CUSTOMSCALECONS  :label "Scale constant"  :entry-width 10 :valuex 15c :min 1 :max 40  \
      :command "click-viewopts 3" :value [expr ($preferences(scale.constant)-1) * 20]
   :slider  CUSTOMSHIFTCONS  :label "Shift constant"  :entry-width 10 :valuex 15c :min 1 :max 40  \
      :command "click-viewopts 6" :value [expr $preferences(shift.constant) * 100]
   :slider  CUSTOMZOOMCONS  :label "Zoom constant"  :entry-width 10 :valuex 15c :min 1 :max 40  \
      :command "click-viewopts 9" :value [expr ($preferences(zoom.constant)-1) * 20]
   :slider  CUSTOMVIEWCONS  :label "View constant"  :entry-width 10 :valuex 15c :min 1 :max 100  \
      :command "click-viewopts 12" :value $preferences(view.constant)

   :message M2 :label "Auto-Repeat" :bold T
   :slider  CUSTOMSCALEFREQ  :label "Scale frequency"  :entry-width 10 :valuex 15c :min 1 :max 20  \
      :command "click-viewopts 1" :value [expr 1000/$preferences(scale.frequency)]
   :slider  CUSTOMSCALEFACT  :label "Scale factor"  :entry-width 10 :valuex 15c :min 1 :max 20  \
      :command "click-viewopts 2" :value [expr ($preferences(scale.factor)-1) * 20]

   :slider  CUSTOMSHIFTFREQ  :label "Shift frequency"  :entry-width 10 :valuex 15c :min 1 :max 20  \
      :command "click-viewopts 4" :value [expr 1000/$preferences(shift.frequency)]
   :slider  CUSTOMSHIFTFACT  :label "Shift factor"  :entry-width 10 :valuex 15c :min 1 :max 20  \
      :command "click-viewopts 5" :value [expr $preferences(shift.factor) * 100]

   :slider  CUSTOMZOOMFREQ  :label "Zoom frequency"  :entry-width 10 :valuex 15c :min 1 :max 20  \
      :command "click-viewopts 7" :value [expr 1000/$preferences(zoom.frequency)]
   :slider  CUSTOMZOOMFACT  :label "Zoom factor"  :entry-width 10 :valuex 15c :min 1 :max 20  \
      :command "click-viewopts 8" :value [expr ($preferences(zoom.factor)-1) * 20]

   :slider  CUSTOMVIEWFREQ  :label "View frequency"  :entry-width 10 :valuex 15c :min 1 :max 20  \
      :command "click-viewopts 10" :value [expr 1000/$preferences(view.frequency)]
   :slider  CUSTOMVIEWFACT  :label "View factor"  :entry-width 10 :valuex 15c :min 1 :max 100  \
      :command "click-viewopts 11" :value $preferences(view.factor)

   :panel CUSTOM1D :padx 5 :pady 5 :layout V
   :layout V; :col-gap 5; :row-gap 5
   :message M3 :label "1D View" :bold T
   :slider  CUSTOMLEFT    :label "left channel:"   :entry-width 10 :valuex 15c :min 0 :max 65536   \
          :command "click-onedviewopts-arg xmin"
   :slider  CUSTOMRIGHT   :label "right channel:"  :entry-width 10 :valuex 15c :min 0 :max 65536   \
          :command "click-onedviewopts-arg xmax"
   :slider  CUSTOMMIN1D   :label "minimum count:"  :entry-width 10 :valuex 15c :min -32768 :max 32768 \
          :command "click-onedviewopts-arg ymin"
   :slider  CUSTOMMAX1D   :label "maximum count:"  :entry-width 10 :valuex 15c :min 0 :max 9999999  \
          :command "click-onedviewopts-arg ymax"
   :layout H
   :next-row T
   :button   CUSTOMAPPLY1D      :label "Apply"        :command "click-onedviewopts-apply"
   :button   CUSTOMGRAB1D       :label "Grab View"    :command "click-onedviewopts-grab"
   :button   CUSTOMSCALE1D      :label "Scale View"   :command "click-onedviewopts-scale"

   :panel CUSTOM2D :padx 5 :pady 5 :layout V
   :layout V; :col-gap 5; :row-gap 5
   :message M2 :label "2D view" :bold T
   :slider  CUSTOMLEFTX  :label "low channel (X):"   :entry-width 10 :valuex 18c :min 0 :max 4096     :command "click-twodviewopts-arg xmin"
   :slider  CUSTOMRIGHTX :label "high channel (X):"  :entry-width 10 :valuex 18c :min 0 :max 4096     :command "click-twodviewopts-arg xmax"
   :slider  CUSTOMLEFTY  :label "low channel (Y):"   :entry-width 10 :valuex 18c :min 0 :max 4096     :command "click-twodviewopts-arg ymin"
   :slider  CUSTOMRIGHTY :label "high channel (Y):"  :entry-width 10 :valuex 18c :min 0 :max 4096     :command "click-twodviewopts-arg ymax"
#   :slider  CUSTOMMIN2D   :label "minimum count (Z):"  :entry-width 10 :valuex 18c :min -1024 :max 1024  :command "click-twodviewopts-arg zmin"
#   :slider  CUSTOMMAX2D   :label "maximum count (Z):"  :entry-width 10 :valuex 18c :min 0 :max 9999999   :command "click-twodviewopts-arg zmax"
   :layout H
   :next-row T
   :button  CUSTOMAPPLY2D      :label "Apply"        :command "click-twodviewopts-apply"
   :button  CUSTOMGRAB2D       :label "Grab View"    :command "click-twodviewopts-grab"

   open-frame [self frame-name].viewopts
   after 1000 click-twodviewopts-grab
   after 1000 click-onedviewopts-grab
}


proc do-advanced-options {} {
   global env frame gallery
   :frame [self frame-name].advancedopts  :label "Advanced Preferences" :show-footer F  \
        :icon-image "@$env(ICONHOME)/spectrum"  \
        :icon-name "Advanced"

   wm protocol [self frame-name].advancedopts  WM_DELETE_WINDOW  "click-quit [self frame-name].advancedopts"

   :panel ADVANCED :padx 5 :pady 5
   :layout V; :col-gap 5; :row-gap 5
   :button  ADVANCED1       :label "Set old 2d display mode"        :command "do-advanced 1"
   :button  ADVANCED2       :label "Set new 2d display mode"        :command "do-advanced 2"
   :button  ADVANCED3       :label "Set debug mode off"             :command "do-advanced 3"
   :button  ADVANCED4       :label "Set debug mode on"              :command "do-advanced 4"
   :button  ADVANCED5       :label "Show configuration"             :command "do-advanced 5"

   open-frame [self frame-name].advancedopts
}


set preferences(spectrum.twodmode) 1     ;#   default is now "new" mode

proc do-advanced opt {
   global gallery preferences spectrumTrace
   switch $opt {
      1  {:spectrum $gallery configure -twodmode 0; set preferences(spectrum.twodmode) 0}
      2  {:spectrum $gallery configure -twodmode 1; set preferences(spectrum.twodmode) 1}
      3  {:spectrum $gallery configure -debug 0; set spectrumTrace 0}
      4  {:spectrum $gallery configure -debug 1; set spectrumTrace 1}
      5  {midas-report "[:spectrum $gallery configure ""]"}
      default {}
   }
}

#   procedures to handle the ALT button actions

set shift_busy 0

set preferences(shift.frequency) 100
set preferences(shift.factor) 0.025

proc click-shift-Press {w which} {
    global shift_busy
  #  select_frame $w
    set shift_busy 1
    repeat-shift $which
}

proc click-shift-Release {} {
    global shift_busy
    set shift_busy 0
}

proc repeat-shift {which} {
    global shift_busy
    global preferences
    global gallery movement
    if {$shift_busy != 1} then {return}
    if {$which == "L"} then {
       if {$movement} then {set s -$preferences(shift.factor)} else {set s $preferences(shift.factor)}
    } else {
       if {$movement} then {set s $preferences(shift.factor)}  else {set s -$preferences(shift.factor)}
    }
    :spectrum $gallery view shift $s
    after $preferences(shift.frequency) "repeat-shift $which"
}

set preferences(shift.constant) 0.25

proc click-shift {which} {
   global gallery preferences movement
   if {$which == "L"} then {
       if {$movement} then {set s -$preferences(shift.constant)} else {set s $preferences(shift.constant)}
   } else {
       if {$movement} then {set s $preferences(shift.constant)}  else {set s -$preferences(shift.constant)}
   }
   :spectrum $gallery view shift $s
}

set preferences(scale.frequency) 100
set preferences(scale.factor) 1.1

set scale_busy 0
set scale_dim 1

proc click-scale-Press {w which} {
    global scale_busy scale_dim gallery
  #  select_frame $w
    set scale_busy 1
    set scale_dim [:spectrum $gallery get dimension]
    repeat-scale $which
}

proc click-scale-Release {} {
    global scale_busy
    set scale_busy 0
}

proc repeat-scale {which} {
    global scale_busy
    global preferences
    global gallery scale_dim
    if {$scale_busy != 1} then {return}

   switch $scale_dim {
      "1" {
             if {$which == "U"} then {set s $preferences(scale.factor)} else {set s [expr 1.0/$preferences(scale.factor)]}
             :spectrum $gallery view scale $s
          }
      "2" {
             change-contour-levels $which REPEAT
          }
      default {}
   }

    after $preferences(scale.frequency) "repeat-scale $which"
}

set preferences(scale.constant) 2

proc click-scale {which} {
   global gallery preferences
   switch [:spectrum $gallery get dimension] {
      "1" {
             if {$which == "U"} then {set s $preferences(scale.constant)} else {set s [expr 1.0 / $preferences(scale.constant)]}
             :spectrum $gallery view scale $s
          }
      "2" {
             change-contour-levels $which
          }
      default {}
   }
}

set preferences(zoom.frequency) 100
set preferences(zoom.factor) 1.1
set preferences(zoom.constant) 2

set zoom_busy 0

proc click-zoom-Press {w which} {
    global zoom_busy
  #  select_frame $w
    set zoom_busy 1
    repeat-zoom $which
}

proc click-zoom-Release {} {
    global zoom_busy
    set zoom_busy 0
}

proc repeat-zoom {which} {
    global zoom_busy preferences
    if {$zoom_busy != 1} then {return}
    click-zoom $which REPEAT
    after $preferences(zoom.frequency) "repeat-zoom $which"
}


set preferences(view.frequency) 100
set preferences(view.factor)  2
set preferences(view.constant) 25

set view_busy 0

proc click-view-Press {w which} {
    global view_busy
  #  select_frame $w
    set view_busy 1
    repeat-view $which
}

proc click-view-Release {} {
    global view_busy
    set view_busy 0
}

proc repeat-view {which} {
    global view_busy preferences
    if {$view_busy != 1} then {return}
    click-view $which REPEAT
    after $preferences(view.frequency) "repeat-view $which"
}


#    procedure to handle interlock on motion events for button 2

set B2_busy 0
proc click-B2-motion {x y} {
   global B2_busy
   if {$B2_busy != 0} then {return}
   set B2_busy 1
   drag $x $y
   set B2_busy 0
}

proc click-B2-release {x y} {
   global B2_busy b2vertex b2tag
   if {$B2_busy == 0} then {drag-end $x $y} else {set b2vertex ""; set b2tag ""}
}

proc click-B2 {W x y} {
   select_frame $W
   drag-start $x $y
}

proc click-B3-Press {W x y} {
   global gallery gate1 gate2

   select_frame $W

   if {[:spectrum $gallery get dimension] != "1"} {
        set-footer "Operation only supported for 1D histograms"; after 1000 clear-footer; return
   }
   set ch [:spectrum $gallery get channel $x $y]
   if {$ch == ""} {set-footer "hit is outside the histogram ($x $y)"; after 1000 clear-footer; return}

   set gate1 "$x $y"
}

proc click-B3-Release {W x y} {
   global gallery gate1 gate2

   if {[:spectrum $gallery get dimension] != "1"} {
        set-footer "Operation only supported for 1D histograms"; after 1000 clear-footer; return
   }
   set ch [:spectrum $gallery get channel $x $y]
   if {$ch == ""} {set-footer "hit is outside the histogram ($x $y)"; after 1000 clear-footer; return}

   set gate2 "$x $y"

   :spectrum $gallery gate set $gate1 $gate2
}

proc click-B1 {W x y} {
   select_frame $W
   decode-hit $x $y
}


#    All standard events return via the click procedure
#    This enables switch of the focus of the active sd window if it has been changed

proc ::enable-event {win name} {
  global frame DisableReSize
#      insert-debug "::enable-event $win $name"

  if {$DisableReSize} {return}

  switch $name {
     Map        -
     Unmap      -
     Visibility -
     Configure  -
     Enter      {click [list $frame :Configure $win]}
     default    {}
  }
}

proc do_configure {w} {
   global heights
   global configuring _classname
   global PreFix frame DisableReSize DisableTopLevel 

      if {$configuring != 0} then {return}
      if {$w != $frame} then {return}

       set f $w
       global $f
       set mode [set ${f}(framemode)]
       if {![info exists heights($mode)]} then {return}

       incr configuring

       if {$DisableTopLevel} then {set g [winfo height .]} else {set g [winfo height $f]}
       set h [set ${f}(geometry)]

       if {$h != $g} then {
          set wh [first $heights($mode)]
          set sh [second $heights($mode)]
          set p S[string range $frame [string length $PreFix] end]
          if {[info exists _classname([string tolower [set p]SPECTRUM])]} then {
              :modify [set p]SPECTRUM :h [expr $g - $wh + $sh]
#insert-debug "configure :modify [set p]SPECTRUM :h [expr $g - $wh + $sh]"
              update
              set ${f}(geometry) $g
          }
       }
       set configuring 0
}

set configuring 0

proc frame-quit {win} {click-quit $win}

proc click arg {
   global PreFix DisableTopLevel
   global env debug frame gallery

   if {$debug} then {insert-debug "click with argument \{$arg\}"}

   set f [lindex $arg 0]
   set c [lindex $arg 1]
   set w [lindex $arg 2]
   set v [lindex $arg 3]

if {$DisableTopLevel} then {

   if {$c == ":quit"} then {
       if {$f == "SD0"} then {click-quit [self frame-name]; return}
   }
}

   if {$c == ":Configure"} then {do_configure $w; return}

   if {$c == ":Map"} then {return}
   if {$c == ":UnMap"} then {return}
   if {$c == ":Visibility"} then {return}
   if {$c == ":Configure"} then {return}
   if {$c == ":Enter"} then {return}

   set f [set PreFix][string range $w 1 1]
   select_frame $f

   if {$c == ":quit"} then {click-quit $f; return}

   switch -glob $w {
        *INVERT      {click-invert}
        *LEFT        {click-shift L}
        *RIGHT       {click-shift R}
        *UP          {click-view U}
        *DOWN        {click-view D}
        *TIMES2      {click-scale U}
        *DIVIDE2     {click-scale D}
        *ZOOMIN      {click-zoom I}
        *ZOOMOUT     {click-zoom O}
        *RESET       {click-reset}
        *REFRESH     {refresh-once}
        *TICK        {click-tick $f}

        *SHOWMODE    {click-showmode [lindex $arg 4]}
        *VIEWMODE    {click-viewmode [lindex $arg 4]}
        *DISPLAYMODE {click-displaymode [lindex $arg 4]}
        *SLICEMODE   {click-slicingmode [lindex $arg 4]}

        *CHANNEL     {click-channel $v}

        *VIEW        {
           switch $v {
              expand          {click-expand}
              "zoom in"       {click-zoom I}
              "zoom out"      {click-zoom O}
              times2          {click-scale U}
              divide2         {click-scale D}
              left            {click-shift L}
              right           {click-shift R}
              up              {click-view U}
              down            {click-view D}
              reset           {click-reset}
              log             {:spectrum $gallery configure -style log}
              linear          {:spectrum $gallery configure -style linear}
              "skip channel 0" {chop-from 1}
              "scale to view" {chop-from}
              undo            {:spectrum $gallery view undo 1}
              redo            {:spectrum $gallery view redo 1}
              retain          {click-retain}
              revert          {click-revert}
              refresh         {}
              Once            -
              Off             -
              0.1             -
              0.25            -
              0.5             -
              1               -
              2               -
              5               -
              10              -
              30              {click-refresh $v}
              preferences           -
              "Save preferences"    {update_preferences}
              "view"                {viewopts}
              "1d colours"          {inform opt start OLCM}
              "2d colours"          {inform opt start CM}
              "2d contours"         {inform opt start SDC}
              Advanced              {do-advanced-options}
              default         {puts stderr "sd: unknown click \{$arg\}"}
           }
        }

        *ARRANGE  {
           switch $v { 
              {"remove all spectra"}     {}
              "remove all spectra"       {:spectrum $gallery spectrum clear}
              "remove current spectrum"  {click-remove}
              "remove all overlaps"      {click-remove-overlap all}
              "remove last overlap"      {click-remove-overlap last}

              rearrange       {:spectrum $gallery spectrum squeeze}
              "example 1d"    {show-builtin 1d}
              "example 2d"    {show-builtin 2d}
              "gallery hardcopy"  {midas-print-object [:path $gallery]}
              "gallery save"      {midas-save-object [:path $gallery]}
              default         {puts stderr "sd: unknown click \{$arg\}"}
           }
        }

        *TAGSFITS  {
           switch $v {
              {"clear tags"}   -
              "clear tags"     {clear-tags}
              "restore tags"   {restore-tags}
              "clear gates"    {clear-gates all}
              "clear gate"     {clear-gates current}
              "clear pointers" {clear-pointers all}
              "clear pointer"  {clear-pointers current}
              "clear fit"      {clear-fit}
              "polygon editor" {}
              "load polygon array"    {click-directory-browser}
              "save polygon array"    {click-directory-browser}
              "select next polygon"   {click-next-polygon}
              "create polygon from tags" {click-add-polygon}
              "delete selected polygon"  {click-delete-polygon selected}
              "delete all polygons"   {click-delete-polygon -1}
              "add vertex"            {click-add-vertex}
              "delete vertex"         {click-delete-vertex}
              "project spectrum"      {click-project-spectrum}
              default         {puts stderr "sd: unknown click \{$arg\}"}
           }
        }

       *ANALYSIS  {
           switch $v {
              integrate                {xhelp; click-integrate {}}
              peakfit                  {xhelp; click-peakfit {}}
              "peakfit options..."     {inform opt start BP}
              brumfit                  {xhelp; click-brumfit {}}
              "brumfit options..."     {inform opt start BF}
              "calibrate..."           {xhelp; click-calibrate {}}
              "peak/total"             {xhelp; click-peak/total {}}
              "max channel counts"     {xhelp; click-max-counts {}}
              "max counts (range)"     {xhelp; click-range-max {}}
              "total counts"           {xhelp; click-total-counts {}}
              "counts (range)"         {xhelp; click-range-counts {}}
              "Co60 peak/total"        {xhelp; click-Co60-peak-to-total {}}
              "Co60 auto-calibration"  {xhelp; click-Co60-auto-calibration {}}
              "Eu152 auto-calibration" {xhelp; click-Eu152-auto-calibration {}}
              "auto-calibration options..."  {inform opt start AC}
              peakfind                 {xhelp; click-peakfind {}}
              "peakfind options..."    {inform opt start PF}
              gainmatch                {xhelp; click-gainmatch {}}
              "gainmatch options..."   {inform opt start GM}
              "create NEO++ calibration files"  {xhelp; make_NEOgainmatch_file}
              "create EB calibration files"  {xhelp; make_EBgainmatch_file}
              plot                     {click-plot {}}
              "plot options..."        {inform opt start SP}
              "save cuts"              {click-save-cuts}
              "check counts"           {click-check-counts {}}
              "fix counts"             {click-fix-counts {}}
              default               {puts stderr "sd: unknown click \{$arg\}"}
           }
        }

        default      {puts stderr "sd: unknown click \{$arg\}"}
   }
}

proc xhelp {} {
   global _suppresshelp

   set _suppresshelp 1
   after 5000  set _suppresshelp 0
}

proc click-onedviewopts-arg {a w v} {
   global gallery onedviewopts
   set onedviewopts($a) $v
   switch $a {
      xmin {
         if [expr $onedviewopts(xmax)-$onedviewopts(xmin)<10] then {
            set onedviewopts(xmin) [expr $onedviewopts(xmax)-10]
         }
      }
      xmax {
         if [expr $onedviewopts(xmax)-$onedviewopts(xmin)<10] then {
            set onedviewopts(xmax) [expr $onedviewopts(xmin)+10]
         }
      }
      ymin {
         if [expr $onedviewopts(ymax)-$onedviewopts(ymin)<10] then {
            set onedviewopts(ymin) [expr $onedviewopts(ymax)-10]
         }
      }
      ymax {
         if [expr $onedviewopts(ymax)-$onedviewopts(ymin)<10] then {
            set onedviewopts(ymax) [expr $onedviewopts(ymin)+10]
         }
      }
   }
   do-onedviewopts-apply
}

proc do-onedviewopts-show {} {
   global onedviewopts
   :modify CUSTOMLEFT  :value $onedviewopts(xmin)
   :modify CUSTOMRIGHT :value $onedviewopts(xmax)
   :modify CUSTOMMIN1D   :value $onedviewopts(ymin)
   :modify CUSTOMMAX1D   :value $onedviewopts(ymax)
}
proc do-onedviewopts-grab {} {
   global gallery onedviewopts
   set r [:spectrum $gallery view get]
   set onedviewopts(xmin) [first $r]
   set onedviewopts(xmax) [second $r]
   set onedviewopts(ymin) [third $r]
   set onedviewopts(ymax) [fourth $r]
}

proc click-onedviewopts-grab {} {
   do-onedviewopts-grab
   do-onedviewopts-show
}

proc do-onedviewopts-apply {} {
   global gallery onedviewopts
   :spectrum $gallery view set $onedviewopts(xmin) $onedviewopts(xmax) $onedviewopts(ymin) $onedviewopts(ymax)
}

proc click-onedviewopts-apply {} {
   global onedviewopts
   if {$onedviewopts(scale)} {
      chop-from
   } {
      do-onedviewopts-apply
   }
}

proc click-onedviewopts-scale {} {
   chop-from
}


# chop-from [args] 
# displays view, scaled to maximum visible channel height 
# chop-from 
# displays current view 
# chop-from start
# displays view from "start" channel to lesser of current view or end of spectrum
# chop-from start end
# displays view from "start" channel to lesser of "end" channel or end of spectrum
# N.B "current view" is the values in onedviewopts. Might not be the values displayed.

proc chop-from args {
   global gallery onedviewopts
   if {[set s [select spectrum]] == ""} then {set-footer "no spectrum selected!!!"; return}
   if {[midas-check-for-temp] == 0} then {return}
# is spectrum a SOAP path
   set sv [inform ex obtain_SOAPService_byName $s]
   if {$sv != ""} {
# build spectrum name in current (temp) directory, replacing /'s in path with _ (e.g. /MemSas/Spect => /tmp/tcl123/MemSAS_Spect)
     set ns [join [tail [split $s /]] _]
     set ts [file join [pwd] $ns]
     catch {eg "(spectrum :path '$ts' !delete)"}
     catch {eval exec copy-soap-spectrum $s $ts} m
   } else {
     set ts $s
   }
   set z [catch {eg "(spectrum :path '$ts' ?spec)"} m]
   if {$z != 0} then {set-footer "$m"; return}
   if {[first $m] != 1} {set-footer "spectrum not 1d"; return}

   if {[llength $args] == 0} {set lim1 -1}
   if {[llength $args] > 0} {set lim1 [first $args]}
   if {[llength $args] > 1} {set lim2 [second $args]} {set lim2 0}
   do-onedviewopts-grab
   if {$lim1 == -1} {set lim1 $onedviewopts(xmin)}
   set onedviewopts(xmin) $lim1
   if {$lim2 != 0} {
     if {$lim2 > [expr [third $m] - 1]} {set lim2 [expr [third $m] - 1]}
     set onedviewopts(xmax) $lim2
   } else {
     set lim2 $onedviewopts(xmax)
   }

   set z [catch {exec max-counts -limit $lim1 $lim2 -spectrum $ts} m]
   if {$sv != ""} {
     catch {eg "(spectrum :path '$ts' !delete)"}
   }
   if {$z != 0} then {set-footer "$m"; return}
   set maxc [third $m]
   for {set p 0; set ymax [expr pow(2,$p)]} {$ymax <= $maxc} {incr p} {set ymax [expr pow(2,$p)]}
   set onedviewopts(ymax) [expr int($ymax)]
   do-onedviewopts-apply
   catch {do-onedviewopts-show}
}

proc click-twodviewopts-arg {a w v} {
   global gallery twodviewopts
   set twodviewopts($a) $v
   switch $a {
      xmin {
         if [expr $twodviewopts(xmax)-$twodviewopts(xmin)<10] then {
            set twodviewopts(xmin) [expr $twodviewopts(xmax)-10]
         }
      }
      xmax {
         if [expr $twodviewopts(xmax)-$twodviewopts(xmin)<10] then {
            set twodviewopts(xmax) [expr $twodviewopts(xmin)+10]
         }
      }
      ymin {
         if [expr $twodviewopts(ymax)-$twodviewopts(ymin)<10] then {
            set twodviewopts(ymin) [expr $twodviewopts(ymax)-10]
         }
      }
      ymax {
         if [expr $twodviewopts(ymax)-$twodviewopts(ymin)<10] then {
            set twodviewopts(ymax) [expr $twodviewopts(ymin)+10]
         }
      }
   }
      :spectrum $gallery view set $twodviewopts(xmin) $twodviewopts(xmax) $twodviewopts(ymin) $twodviewopts(ymax)
}

proc click-twodviewopts-grab {} {
   global gallery twodviewopts
   set r [:spectrum $gallery view get]
   set twodviewopts(xmin) [first $r]
   set twodviewopts(xmax) [second $r]
   set twodviewopts(ymin) [third $r]
   set twodviewopts(ymax) [fourth $r]
   :modify CUSTOMLEFTX  :value $twodviewopts(xmin)
   :modify CUSTOMRIGHTX :value $twodviewopts(xmax)
   :modify CUSTOMLEFTY  :value $twodviewopts(ymin)
   :modify CUSTOMRIGHTY :value $twodviewopts(ymax)
}

proc click-twodviewopts-apply {} {
   global gallery twodviewopts
   :spectrum $gallery view set $twodviewopts(xmin) $twodviewopts(xmax) $twodviewopts(ymin) $twodviewopts(ymax)
}

proc click-viewopts {which w v} {
    global preferences
    switch $which {
        1     {set preferences(scale.frequency) [expr 1000/$v]}
        2     {set preferences(scale.factor)    [expr 1.0 + ($v/20.0)]}
        3     {set preferences(scale.constant)  [expr 1.0 + ($v/20.0)]}

        4     {set preferences(shift.frequency) [expr 1000/$v]}
        5     {set preferences(shift.factor)    [expr $v/100.0]}
        6     {set preferences(shift.constant)  [expr $v/100.0]}

        7     {set preferences(zoom.frequency) [expr 1000/$v]}
        8     {set preferences(zoom.factor)    [expr 1.0 + ($v/20.0)]}
        9     {set preferences(zoom.constant)  [expr 1.0 + ($v/20.0)]}

        10     {set preferences(view.frequency) [expr 1000/$v]}
        11     {set preferences(view.factor)    $v}
        12     {set preferences(view.constant)  $v}
    }
}


proc start-for-sb {} {
    global started frame
    if {$started == 0} then {
       start
    } else {
       open-frame $frame
    }
}

proc start args {
   global started env frame frametitle
   global preferences spectrum_height cuts_height DisableTopLevel DisableReSize
   global tcl_platform

   if {$started == 0} then {

   if {$tcl_platform(platform) != "windows"} then {

#    check for availability of suitable visual

      set d [winfo depth .]
      if {$d != 8 && $d != 24 && $d != 32} then {
           midas-warning "Display depth is $d. This cannot be used for 2D displays.\n\
               You must reconfigure your display software to run in either 8 bit mode (PseudoColor 8),\
               24 bit mode (TrueColor 24) or 32 bit mode (DirectColor 32)."
      }
    }
      catch {wm withdraw .}

      set f [new-sd-frame]
      global $f
      set ${f}(framemode) "spectrum"
      change-frame $f
      set frametitle "Displaying Spectra ($frame)"

      set_default_options
      load_preferences

      set spectrum_height $preferences(spectrum.spectrum_height)
      set cuts_height $preferences(spectrum.cuts_height)
      set DisableTopLevel $preferences(spectrum.DisableTopLevel)
      set DisableReSize $preferences(spectrum.DisableReSize)


set DisableTopLevel 0
set DisableReSize 0


      update_preferences

#    initialise options (via opt task)

      inform opt initialise-colours
      inform opt initialise-linecolours

      init-sd-frame $spectrum_height
      fix-ticks
   
      set started 1
   } else {

      open-frame $frame
   }
}

proc click-quit f {
   global frames started
   destroy-frame $f
   if {![member $f $frames]} then return
   delete-sd-frame $f
   if {[first $frames] == ""} then {
      set started 0
   } else {
      click-tick [first $frames]
   }
}

proc finish args {
   global frames
   foreach f $frames {click-quit $f}
}


proc set_default_options {} {
      global backpeakoptions peakfindoptions specplotoptions gaincalcoptions autocalibrateoptions
      global movement fittags

      set backpeakoptions(backtype) slopesmoo
      set backpeakoptions(peaktype) gauss
      set backpeakoptions(weight)   count
      set backpeakoptions(shape)    differen
      set backpeakoptions(peaks)    1
      set backpeakoptions(order)    1

      set peakfindoptions(low-chan) 0
      set peakfindoptions(high-chan) 4095
      set peakfindoptions(fwhm)     5
      set peakfindoptions(accept)   10

      set specplotoptions(low-chan) 0
      set specplotoptions(high-chan) 4095
      set specplotoptions(scale)    200
      set specplotoptions(strips)   2
      set specplotoptions(norm)     linear
      set specplotoptions(peaks)     0
      set specplotoptions(gains)    [list [list 0 4096 0 70000]]
      set specplotoptions(paper)    A4

      set gaincalcoptions(source)   "0"
      set gaincalcoptions(voverc)   "0.0"
      set gaincalcoptions(energy1)  0
      set gaincalcoptions(width1)   ""
      set gaincalcoptions(energy2)  ""
      set gaincalcoptions(width2)   0
      set gaincalcoptions(mapfile)  ""
      set gaincalcoptions(outputfile) ""

      set autocalibrateoptions(low-chan)   1
      set autocalibrateoptions(high-chan)  4095
      set autocalibrateoptions(Eu-accept)  20
      set autocalibrateoptions(Eu-minpeaks) 7
      set autocalibrateoptions(Eu-maxpeaks) 11
      set autocalibrateoptions(Co-minpeaks) 2
      set autocalibrateoptions(Co-maxpeaks) 2


      set movement [get-eg-option spectrum-movement]

      set fittags {}
}

proc new-sd-frame {} {
   global PreFix frames showmode slicingmode
   set i 0
   while {[lsearch $frames [set PreFix]$i] >= 0} {incr i}
   set f [set PreFix]$i
   lappend frames $f
   global $f
   set  ${f}(cuts)  ""
   set  ${f}(cutptrs) {}
   set  ${f}(cutframe) ""
   set  ${f}(framemode) ""
   set  ${f}(geometry) ""
   set  ${f}(showmode) $showmode
   set  ${f}(slicingmode) $slicingmode
   return $f
}

proc delete-sd-frame f {
   global frames
   set i [lsearch $frames $f]
   set frames [lreplace $frames $i $i]
}

#    rebuild the colourmaps to overcome the behaviour (bug?) of tcl8.5/8.6
#    the 1st element might have become a list itself ( {{#01} #12 #23 #...} )

proc init-sd-frame {h} {
   global gallery colourmap overlapcolourmap
   mksd $h
   if {$colourmap != ""} then {:spectrum $gallery configure -colourmap $colourmap}
   if {$overlapcolourmap != ""} then {:spectrum $gallery configure -overlapcolourmap $overlapcolourmap}
   set-contour-levels
}

#    procedure called by the Clone event to start another sd window
proc click-another {} {
   global frame frametitle
   global spectrum_height
   set f [new-sd-frame]
   global $f
   set ${f}(framemode) "spectrum"
   change-frame $f
   set frametitle "Displaying Spectra ($frame)"
   init-sd-frame $spectrum_height
   fix-ticks
}

#    procedures for the "active frame" tick indicator
#    also procedures used to select the "active frame"

#    We need to be cautious because of problems when a Cuts frame is
#    created by draging using button 2. There can be many mouse events
#    while the frame is still being created

proc fix-ticks {} {
   global PreFix frame frames
   global B2_busy
   if {$B2_busy != 0} then {return}
   foreach f $frames {
      set p [string range $f [expr [string length $PreFix] - 1] end]
      if {$f == $frame} then {set vl 1} else {set vl 0}
      catch {:modify [set p]TICK :value $vl}
   }
   open-frame $frame
}

proc click-tick v {
   change-frame $v
   fix-ticks
}

proc change-frame {f} {
   global PreFix frame gallery debug
   set frame $f
   set gallery S[string range $frame [string length $PreFix] end]
   if {$debug} then {insert-debug "sd: change-frame $frame $gallery"}
}

#    procedure called on all mouse events to check for change of active sd window
proc select_frame p {
   global PreFix frame
   set f [string range $p 0 [string length $PreFix]]

#    next 2 lines needed to prevent spectrum galley windows becoming confused
   raise $f
   update

   if {$f == $frame} then {return}

   change-frame $f
   fix-ticks

####   global DisableTopLevel
####   if {!$DisableTopLevel} {return}

   global gallery
   :spectrum $gallery spectrum refresh
   :spectrum $gallery refresh tags
}

proc build-specname {s args} {

# is spectrum a EG or SOAP path
   if {[inform ex obtain_SOAPService_byName $s] == {}} {
# standard EGlib spectrum name (direct or SAS)
     if {[llength $args] != 0} {set s [join [list [first $args] $s] :]}
   }
    return "1 $s"
}



#   procedures to display a spectrum

#    display 1 or more spectra from the list in $args
#    64 is the most that can be displayed - so threshold 65 effectively means don't deferupdate
set deferThreshold 65

proc show args {
   global deferThreshold
   global DisableTopLevel
   global env started frame frames gallery linestyleoptions
   global spectrumTrace debug
   global $frame

   if {$started != 0} then {
       if {$DisableTopLevel} then {open-frame $frame; clear-footer} else {
           if {[wm state $frame] != "normal"} then {open-frame $frame} else {raise $frame}
           clear-footer
       }
   } else {start; global $frame}

   set fm [set ${frame}(framemode)]
   if {$fm != "spectrum"} then {
      set msg "Current window $frame is not for displaying spectra - use = $fm\n\
                     Please select a Displaying Spectra window"

      set z [midas_dialog [self frame-name].warningDialog "Warning for [self frame-title]" \
              "$msg" "warning" 0 "continue" "please create window"]

      if {$z != 1} then {return}
      click-another
   }
   set sm [set ${frame}(showmode)]
   if {$sm == "new"} then {:spectrum $gallery spectrum clear}
   if {[llength $args] == 1} {set args [first $args]}
   if {[llength $args] > $deferThreshold} {
      set-footer "Please wait: your spectra are being drawn in the background"
      set-busy
      catch {:spectrum $gallery configure -deferupdate 1}
      set t11 [clock clicks -milliseconds]
   }

   foreach s $args {
# is spectrum a EG or SOAP path
      if {[inform ex obtain_SOAPService_byName $s] == {}} {
# standard EGlib spectrum name (direct or SAS)
         set s [join [list "EGlib" $s] :]
      }
      if {$sm == "overlap"} then {
         :spectrum $gallery spectrum overlap $s
      } else {
         :spectrum $gallery spectrum add $s
      }
      :spectrum $gallery spectrum linestyle $linestyleoptions(spectrum)
   }
   if {[llength $args] > $deferThreshold} {
      set t12 [clock clicks -milliseconds]
      catch {:spectrum $gallery configure -deferupdate 0}
      clear-busy
      clear-footer
      insert-debug "[llength $args] spectra drawn in [expr $t12 - $t11]ms ([expr ($t12 - $t11) / [llength $args]]ms each)"
   }
}

proc unshow args {
      set z [catch {eg "(spectrum :name 'xxx' ?spec)"} m]
}

#    procedures which action simple requests to change spectrum view
#    these are called via the VIEW menu and directly from the "shortcut" buttons
#    and the "special" bindings

proc click-reset {} {
   global gallery
   :spectrum $gallery view reset
   switch [:spectrum $gallery get dimension] {
       "1"     {}
       "2"     {inform opt SDC.reset-contours}
       default {}
   }
}

#    procedure called by UP/DOWN keys
proc click-updown {action} {
   global gallery
   switch [:spectrum $gallery get dimension] {
      "1" {click-scale $action}
      "2" {click-view $action}
      default {}
   }
}

set MAX1DCOLOURS 16
set MAX2DCOLOURS 32

#    procedure to invert colour scheme of the spectrum gallery
proc click-invert {} {
   global gallery MAX1DCOLOURS MAX2DCOLOURS
#    we assume that background and forground of the gallery are the same colour
   set clr [last [[:path $gallery] configure -background]]
   set clr [invertcolour $clr]
   [:path $gallery] configure -background $clr
   [:path $gallery] configure -foreground $clr
#    text colour
   set clr [last [[:path $gallery] configure -textcolour]]
   set clr [invertcolour $clr]
   [:path $gallery] configure -textcolour $clr
   set clr [last [[:path $gallery] configure -activetextcolour]]
   set clr [invertcolour $clr]
   [:path $gallery] configure -activetextcolour $clr
#    box colour
   set clr [last [[:path $gallery] configure -boxcolour]]
   set clr [invertcolour $clr]
   [:path $gallery] configure -boxcolour $clr
   set clr [last [[:path $gallery] configure -activeboxcolour]]
   set clr [invertcolour $clr]
   [:path $gallery] configure -activeboxcolour $clr
#    tag colour
   set clr [last [[:path $gallery] configure -tagcolour]]
   set clr [invertcolour $clr]
   [:path $gallery] configure -tagcolour $clr
#    gate colour
   set clr [last [[:path $gallery] configure -gatecolour]]
   set clr [invertcolour $clr]
   [:path $gallery] configure -gatecolour $clr
#    poly colour
   set clr [last [[:path $gallery] configure -polycolour]]
   set clr [invertcolour $clr]
   [:path $gallery] configure -polycolour $clr
#    pointer colour
   set clr [last [[:path $gallery] configure -pointercolour]]
   set clr [invertcolour $clr]
   [:path $gallery] configure -pointercolour $clr
#    1d colours
   set clrmap [last [[:path $gallery] configure -overlapcolourmap]]
   for {set i 0} {$i < $MAX1DCOLOURS} {incr i} {
      set clr [lindex $clrmap $i]
      set clr [invertcolour $clr]
      set clrmap [lreplace $clrmap $i $i $clr]
   }
   [:path $gallery] configure -overlapcolourmap $clrmap
#    2d colours
   set clrmap [last [[:path $gallery] configure -colourmap]]
   for {set i 0} {$i < $MAX2DCOLOURS} {incr i} {
      set clr [lindex $clrmap $i]
      set clr [invertcolour $clr]
      set clrmap [lreplace $clrmap $i $i $clr]
   }
   [:path $gallery] configure -colourmap $clrmap
}

proc invertcolour clr {
   if {$clr == "black" || $clr == "#000000"} then {return "white"}
   if {$clr == "white" || $clr == "#ffffff"} then {return "black"}
   set rgb [winfo rgb . $clr]
   set r [expr [first $rgb]/0x100]
   set r [expr 255 - $r]
   set g [expr [second $rgb]/0x100]
   set g [expr 255 - $g]
   set b [expr [third $rgb]/0x100]
   set b [expr 255 - $b]
   set clr [format "#%02x%02x%02x" $r $g $b]
   return $clr
}

proc click-view {action args} {
   global preferences movement gallery
   set r [:spectrum $gallery view get]
   set xmin [first $r]
   set xmax [second $r]
   set ymin [third $r]
   set ymax [fourth $r]

   if {[llength $args] == 0} then {set s1 $preferences(view.constant)} else {set s1 $preferences(view.factor)}

   set fract [expr [expr $ymax - $ymin] * $s1 / 100]
   if {$fract == 0} then {set fract 1}
   if {$action == "U"} then {
       if {$movement} then {set s "+"} else {set s "-"}
   } else {
       if {$movement} then {set s "-"} else {set s "+"}
   }
   set ymin [expr $ymin $s $fract]
   set ymax [expr $ymax $s $fract]
   :spectrum $gallery view set $xmin $xmax $ymin $ymax
}

#   next 2 procs used by opt
proc get-zoom-region {} {global gallery; return [:spectrum $gallery view get]}
proc zoom-to-region {region} {global gallery; :spectrum $gallery view set [first $region] [second $region] [third $region] [fourth $region]}

proc get-spectrum-attribute att {global gallery; return [:spectrum $gallery spectrum $att]}
proc get-spectrum-title {} {return [get-spectrum-attribute title]}
proc get-spectrum-name {} {return [get-spectrum-attribute name]}

proc set-contour-levels {} {
   global  gallery
   :spectrum $gallery configure -contours [inform opt set preferences(SDC.contours)]
}

#    action is up or down

proc change-contour-levels {action args} {
   global gallery preferences

   set base [inform opt set preferences(SDC.base)]
   set mode [inform opt set preferences(SDC.mode)]
   set levels [inform opt set preferences(SDC.contours)]

   set newlevels ""
   set prev [first $levels]
   if {[llength $args] == 0} then {set s $preferences(scale.constant)} else {set s $preferences(scale.factor)}
   foreach l $levels {
      if {$action == "U"} then {
         set new [expr [expr $l - $prev] * $s]
      } else {
         set new [expr [expr $l - $prev +1] / $s]
      }
      lappend newlevels $new
   }
   set levels $newlevels
   :spectrum $gallery configure -contours $levels

   inform opt "set preferences(SDC.base) $base"
   inform opt "set preferences(SDC.mode) $mode"
   inform opt "set preferences(SDC.contours) [list $levels]"
}

proc click-zoom {action args} {
#    action is I (In) or O (Out)
   global gallery preferences
   global debug
   set view [:spectrum $gallery view get]
   if {[llength $args] == 0} then {set s1 $preferences(zoom.constant)} else {set s1 $preferences(zoom.factor)}
   set s2 [expr $s1 * 2]
   set s3 [expr $s1 / 2]
   switch [:spectrum $gallery get dimension] {
      "1" {
          set xrange [expr [second $view] - [first $view] + 1]
          set c [:spectrum $gallery tag get 1]
          if {$c==""} then {
             set x1 [first $view]
             if {$action == "I"} then {
                set x2 [expr [first $view] + [expr $xrange/$s1] -1]
             } else {
                set x2 [expr [second $view] + [expr $xrange*$s3]]
             }
          } else {
             if {$action == "I"} then {
                set x1 [expr $c - [expr $xrange/$s2]]
                set x2 [expr $c + [expr $xrange/$s2]]
             } else {
                set x1 [expr $c - [expr $xrange*$s3]]
                set x2 [expr $c + [expr $xrange*$s3]]
             }
#            :spectrum $gallery tag pop 1
          }
          :spectrum $gallery view set $x1 $x2 [third $view] [fourth $view]
      }
      "2" {
          set xrange [expr [second $view] - [first $view] + 1]
          set yrange [expr [fourth $view] - [third $view] + 1]
          set c [:spectrum $gallery tag get2d 1]
          if {$c == ""} then {
             set x1 [first $view]
             set y1 [third $view]
             if {$action == "I"} then {
                set x2 [expr [first $view] + [expr $xrange/$s1] - 1]
                set y2 [expr [third $view] + [expr $yrange/$s1] - 1]
             } else {
                set x2 [expr [second $view] + [expr $xrange*$s3]]
                set y2 [expr [fourth $view] + [expr $yrange*$s3]]
             }
          } else {
             if {$action == "I"} then {
                set x1 [expr [first $c] - [expr $xrange/$s2] + 4]
                set x2 [expr [first $c] + [expr $xrange/$s2] - 4]
                set y1 [expr [second $c] - [expr $yrange/$s2] + 4]
                set y2 [expr [second $c] + [expr $yrange/$s2] - 4]
             } else {
                set x1 [expr [first $c] - [expr $xrange*$s3] + 4]
                set x2 [expr [first $c] + [expr $xrange*$s3] - 4]
                set y1 [expr [second $c] - [expr $yrange*$s3] + 4]
                set y2 [expr [second $c] + [expr $yrange*$s3] - 4]
             }
#            :spectrum $gallery tag pop 1
          }
          if {$debug} then {puts "zoom $action view <$c> $x1 $x2 $y1 $y2"}
          :spectrum $gallery view set $x1 $x2 $y1 $y2
      }
      default {}
   }
}

#    text has been entered into the channel entry widget

proc click-channel t {
    global gallery frame PreFix

    set l [split $t ,]
    set n [llength $l]
    switch $n {
      2 {
        set lx [first $l]
        if {[string index $lx 0] == "="} {
           set calib 1
           set x [string range $lx 1 end]
        } else {
           set calib 0
           set x $lx
        }
        set z [scan $x "%d" cx]
        if {$z != 1} then {set-footer "not a channel number!!! (x=$x)"; return}
        if {$calib} then {set cx [:spectrum $gallery get decalibrationx $cx]}

        set ly [second $l]
        if {[string index $ly 0] == "="} {
           set calib 1
           set x [string range $ly 1 end]
        } else {
           set calib 0
           set x $ly
        }
        set z [scan $x "%d" cy]
        if {$z != 1} then {set-footer "not a channel number!!! (y=$x)"; return}
        if {$calib} {set cy [:spectrum $gallery get decalibrationy $cy]}
        :spectrum $gallery tag push2d $cx $cy
      }
      1 {
        if {[string index $l 0] == "="} {
           set calib 1
           set x [string range $l 1 end]
        } else {
           set calib 0
           set x $l
        }
        set z [scan $x "%d" c]
        if {$z != 1} then {set-footer "not a channel number!!! ($x)"; return}
        if {$calib} {set c [:spectrum $gallery get decalibrationx $c]}
        make-tag $c
      }
      default {set-footer "Format error ... xxx xxx,yyy (=xxx/=yyy for calib input)"}
   }
   set p S[string range $frame [string length $PreFix] end]
   :modify [set p]CHANNEL :value ""
}

proc click-retain {} {
   global  gallery viewring viewnext
   set view [:spectrum $gallery view get]
   if {$view == {}} then {return}
   if {[llength $viewring] < 4} then {
      lappend viewring $view
   } else {
      set viewring [lreplace $viewring 0 0]
      lappend viewring $view
   }
   set viewnext [expr [llength $viewring] -1]
}

proc click-revert {} {
   global  gallery viewring viewnext
   if {$viewring == {}} then {return}
   set view [lindex $viewring $viewnext]
   set left [lindex $view 0]
   set right [lindex $view 1]
   if {$viewnext == 0} then {
      set viewnext [expr [llength $viewring] -1]
   } else {
      incr viewnext -1
   }
   :spectrum $gallery view set $left $right
}


#    2 shortcuts bound to button 2
proc double2 {x y} {
   global gallery
   :spectrum $gallery tag set $x $y
   click-zoom I
}

proc shiftdouble2 {x y} {
   global gallery
   :spectrum $gallery tag set $x $y
   click-zoom O
}

#    procedures called from the Arrange menu

proc show-builtin {d} {
   global gallery linestyleoptions
   global frame
   global $frame
   set fm [set ${frame}(framemode)]
   if {$fm != "spectrum"} then {
      midas-warning "Current window $frame is not for displaying spectra - use = $fm\n   \
                     Please select a Displaying Spectra window"
      return
   }
   set s _inbuilt[set d]_
   set sm [set ${frame}(showmode)]
   if {$sm == "new"} {:spectrum $gallery spectrum clear}
   if {$sm == "overlap"} then {
      :spectrum $gallery spectrum overlap $s
   } else {
      :spectrum $gallery spectrum add $s
   }
   :spectrum $gallery spectrum linestyle $linestyleoptions(spectrum)
}

proc click-remove {} {
   global gallery
   :spectrum $gallery spectrum remove
   :spectrum $gallery spectrum squeeze
}

#    procedures called from the Tags and Fits menu

proc clear-gates {which} {
   global gallery
   if {$which == "all"} then {
      :spectrum $gallery gate pop -1
   } else {
      :spectrum $gallery gate pop 1
   }
}

proc clear-pointers {which} {
   global gallery
   if {$which == "all"} then {
      :spectrum $gallery pointer pop -1
   } else {
      :spectrum $gallery pointer pop 1
   }
}

proc clear-tags {} {
   global gallery
   :spectrum $gallery tag pop -1
}

proc clear-fit {} {
    global gallery
 :spectrum $gallery spectrum removelastoverlap
 :spectrum $gallery spectrum removelastoverlap
}
proc clear-fits {} {}

proc click-remove-overlap {which} {
   global gallery
   if {$which == "all"} then {
      :spectrum $gallery spectrum removealloverlaps
   } else {
      :spectrum $gallery spectrum removelastoverlap
   }
}

proc restore-tags {} {
   global fittags
   clear-tags
   foreach t $fittags {make-tag $t}
}

proc make-tag c {
   global gallery
   :spectrum $gallery tag push $c
}

proc make-pointer c {
   global gallery
   :spectrum $gallery pointer push $c
}


#    procedures called directly by procedure click to action
#    mouse interrupts


proc decode-hit {x y} {
    global gallery

    set d [:spectrum $gallery get dimension]
    if {$d == 1} {
        set ch [:spectrum $gallery get channel $x $y]
        if {$ch == ""} {set-footer "hit is outside the histogram ($x $y)"; after 1000 clear-footer; return}
        set h [:spectrum $gallery tag near $x $y]
        if {$h != ""} then {return}    ;#  already tag at this point
    }

    if {$d == 2} {
        set chX [:spectrum $gallery get channelx $x $y]
        set chY [:spectrum $gallery get channely $x $y]
        if {$chX == "" || $chY == ""} {set-footer "hit is outside the histogram ($x $y)"; after 1000 clear-footer; return}
        set h [:spectrum $gallery gate vertex hit $x $y]
        if {$h != ""} then {
           select-poly-by-number [first $h]
           return
        }
    }

    :spectrum $gallery tag set $x $y
}

proc click-expand {} {
   global gallery
   switch [:spectrum $gallery get dimension] {
      "1" {expand-1d}
      "2" {expand-2d}
      default {}
   }
}

proc expand-1d {} {
   global gallery
   set tags [get-tags 2]
   if {[llength $tags] < 2} then {set-footer "you haven't set any expand tags!!!"; return}
   set currentchannel [second $tags]
   set lastchannel [first $tags]
   if {$currentchannel == $lastchannel} then {return}
   :spectrum $gallery tag pop 2
   :spectrum $gallery view set $lastchannel $currentchannel
}

proc expand-2d {} {
    global gallery
    set tags [get-tags2d 2]
    if {[llength $tags] < 4} then {set-footer "you haven't set enough expand tags---two needed!!!"; return}
    set xs [lsort -integer [list [first $tags] [third $tags]]]
    set ys [lsort -integer [list [second $tags] [fourth $tags]]]
    set w [expr [second $xs]-[first $xs]]
    set h [expr [second $ys]-[first $ys]]
    if {$w>0 && $h>0} then {
        :spectrum $gallery view set [first $xs] [second $xs] [first $ys] [second $ys]
    }
    :spectrum $gallery tag pop 2
}

proc get-tags n {
   global gallery
   switch [:spectrum $gallery get dimension] {
      "1" {
         set tags [:spectrum $gallery tag get $n]
      }
      "2" {
         set tags [:spectrum $gallery tag get2d $n]
      }
      default {set tags {}}
   }
   return $tags
}

proc get-tags2d n {
   global gallery
   set z [catch {set b [select box]}]
   if {$z == 0} then {
      set tags [:spectrum $gallery tag get2d $n]
   } else {
      set tags {}
   }
   return $tags
}


#    procedures to handle the colour maps
#    these procedures are called from the opt task
#    rebuild the colourmaps to overcome the behaviour (bug?) of tcl8.5/8.6
#    the 1st element might have become a list itself ( {{#01} #12 #23 #...} )

proc set-colourmap {s} {
   global gallery colourmap
   set colourmap $s
   catch {:spectrum $gallery configure -colourmap $colourmap}
}

proc set-linecolourmap {s} {
   global gallery overlapcolourmap
   set overlapcolourmap $s
   catch {:spectrum $gallery configure -overlapcolourmap $overlapcolourmap}
}

#    procedures to handle refresh submenu of View menu
#    these are used for auto-refresh of online spectra

proc refresh-once {} {
   global gallery
######   if {![winfo ismapped .[string range $gallery 1 3]]} then {return}
   set-footer "refreshing display..."
   :spectrum $gallery spectrum refresh
   :spectrum $gallery refresh tags
   clear-footer
}

proc do-refresh {} {
   global refreshafter refreshperiod
   if {$refreshafter == ""} then {return}
   refresh-once
   after $refreshperiod do-refresh
}

proc stop-refresh {} {
   global refreshafter
   set refreshafter ""
}

proc start-refresh {} {
   global refreshafter refreshperiod
   set refreshafter [after $refreshperiod do-refresh]
}

proc click-refresh {m} {
   global refreshperiod
   if {$m == "Off"}  then {stop-refresh; return}
   if {$m == "Once"} then {refresh-once; return}
   stop-refresh
   set refreshperiod [expr int (1000*$m)]
   refresh-once
   start-refresh
}

#    procedures to handle the option menus

proc click-showmode opt {
   global frame
   global $frame
   set ${frame}(showmode) $opt
}

proc click-viewmode opt {
   global viewmode gallery
   set viewmode $opt
   :spectrum $gallery configure -select $viewmode
}

proc click-displaymode opt {
   global displaymode gallery
   set displaymode $opt
   :spectrum $gallery configure -style $displaymode
}

proc click-slicingmode opt {
   global frame
   global $frame
   set ${frame}(slicingmode) [second $opt]
}

#    procedures to handle "drag" functions
#    envokes by bindings on Button 2


proc drag {x y} {
   global gallery debug b2tag b2vertex
   if {$debug} then {puts stdout "drag $x $y"}
   switch [:spectrum $gallery get dimension] {
      "1" {

            set ch [:spectrum $gallery get channel $x $y]
            if {$ch == ""} {set-footer "hit is outside the histogram ($x $y)"; after 100 clear-footer; return}

           drag-tag $x $y
           drag-on-spectrum1d $x $y
      }
      "2" {

            set chX [:spectrum $gallery get channelx $x $y]
            set chY [:spectrum $gallery get channely $x $y]
            if {$chX == "" || $chY == ""} {set-footer "hit is outside the histogram ($x $y)"; after 1000 clear-footer; return}

          if {$b2tag != ""} then {drag-tag $x $y}
          if {$b2vertex != ""} then {drag-vertex $x $y}
          drag-on-spectrum2d $x $y
      }
      default {}
   }
}

proc drag-start {x y} {
    global frame gallery b2tag b2vertex
    global $frame
    switch [:spectrum $gallery get dimension] {
        "1" {
            set ch [:spectrum $gallery get channel $x $y]
            if {$ch == ""} {set-footer "hit is outside the histogram ($x $y)"; after 100 clear-footer; return}
            set b2tag [:spectrum $gallery tag near $x $y]
            if {$b2tag != ""} then {return}
            set b2tag [:spectrum $gallery tag set $x $y]
        }
        "2" {
            set chX [:spectrum $gallery get channelx $x $y]
            set chY [:spectrum $gallery get channely $x $y]
            if {$chX == "" || $chY == ""} {set-footer "hit is outside the histogram ($x $y)"; after 1000 clear-footer; return}
            set b2tag [:spectrum $gallery tag near $x $y]
            set h [:spectrum $gallery gate vertex hit $x $y]
            if {$h != ""} then {
#    have hit on a vertex - activation of a vertex is a toggle operation
#    and we must ensure that we leave in the active state
#    things fail if we are starting to drag an already active vertex because
#    we have just deactivated it
               if {[fifth $h] == 0} then {:spectrum $gallery gate vertex hit $x $y}
               select-poly-by-number [first $h]
               set b2vertex "$x $y"
            }
            set lm [set ${frame}(slicingmode)]
            if {$lm == "on"} then {
               global frame; global $frame frames
               set cf [set ${frame}(cutframe)]
               if {$cf != "" && [lsearch $frames $cf] >= 0} then {open-frame $cf}
            }
        }
        default {}
    }
}

proc drag-end {x y} {
    global gallery b2vertex b2tag
    switch [:spectrum $gallery get dimension] {
        "1" {
        }
        "2" {
            if {$b2vertex != ""} then {
                :spectrum $gallery gate vertex hit [first $b2vertex] [second $b2vertex]
                set b2vertex ""
            }
            if {$b2tag != ""} then {set b2tag ""}
        }
        default {}
    }
}

proc drag-on-spectrum1d {x y} {
   global gallery
   set ch [:spectrum $gallery get channel $x $y]
   if {$ch == ""} {set-footer "hit is outside the histogram ($x $y)"; after 1000 clear-footer; return}
   set co [:spectrum $gallery get count $x $y]
   set ca [:spectrum $gallery get calibration $x $y]
   set f "channel: $ch, count: $co"
   if {"$ca" != ""} then {append f ", calibration: $ca"}
   set-footer $f
}

proc drag-on-spectrum2d {x y} {
   global frame
   global $frame
   global gallery
   set xc [:spectrum $gallery get channelx $x $y]
   set yc [:spectrum $gallery get channely $x $y]
   if {"$xc" == "" || "$yc" == ""} then {set-footer "hit is outside the histogram ($x $y)"; after 1000 clear-footer; return}
   set co [:spectrum $gallery get count $x $y]
   set cx [:spectrum $gallery get calibrationx $x $y]
   set cy [:spectrum $gallery get calibrationy $x $y]
   set f "channel: ($xc,$yc) count: $co"
   if {"$cx"!="" || "$cy"!=""} then {
      append f ", calibration: "
      if {"$cx" != ""} then {append f "$cx"}  else {append f "???"}
      if {"$cy" != ""} then {append f ",$cy"} else {append f ",???"}
   }
   set-footer $f
   set lm [set ${frame}(slicingmode)]
   if {$lm == "on"} then { cut $xc $yc }
}

proc drag-tag {x y} {
    global gallery
    :spectrum $gallery tag move $x $y
}

proc drag-vertex {x y} {
    global gallery b2vertex
    :spectrum $gallery gate vertex set $x $y
    set b2vertex "$x $y"
}

proc cut {x y} {
   global frame frames frametitle gallery
   global cuts_height

#    check coordinates are inside the spectrum
   set info [:spectrum $gallery spectrum info]
   if {$x < [first $info]} {return}
   if {$y < [third $info]} {return}
   if {$x > [second $info]} {return}
   if {$y > [fourth $info]} {return}

   set matrix [:spectrum $gallery spectrum current]

   set titlex "[select spectrum]@x=${x}"
   set titley "[select spectrum]@y=${y}"
# The following 2 lines ARE correct
# the cut@x is onto the y-axis so it needs the y calibration & vice versa
   set calx [:spectrum $gallery spectrum calibrate y]
   set caly [:spectrum $gallery spectrum calibrate x]

#  now proceed to display the result of the cut

   set frameSAVE $frame

   global $frame
   set cf [set ${frame}(cutframe)]
   if {$cf == "" || [lsearch $frames $cf] < 0} then {
      set cf [new-sd-frame]
      global $cf
      set ${frame}(cutframe) $cf
      set ${cf}(cutframe) $frame
      set ${cf}(framemode) "cut"
      change-frame $cf
      set frametitle "Displaying 2d Cuts/Projections ($frame)"
      init-sd-frame  $cuts_height
   } else {
      global $cf
      change-frame $cf
   }

   set cuts [set ${cf}(cutptrs)]

   set cutx [first $cuts]
   set cuty [second $cuts]

   if {$cutx!=""} then {
      :spectrum $gallery spectrum current $cutx
      set oldtitlex [:spectrum $gallery spectrum title]
      if {"$oldtitlex" != "$titlex (unsaved)"} then {
         set cutx ""
         set cuty ""
      }
   }

   if {$cutx==""} then {
      :spectrum $gallery spectrum clear
      :spectrum $gallery spectrum add-dataset cutx [third $info] [fourth $info]
      set cutx [:spectrum $gallery spectrum current]
   }
   if {$cuty==""} then {
      :spectrum $gallery spectrum add-dataset cuty [first $info] [second $info]
      set cuty [:spectrum $gallery spectrum current]
   }

   :spectrum $gallery cut x $x $matrix $cutx
   :spectrum $gallery spectrum current $cutx
   :spectrum $gallery spectrum title "$titlex (unsaved)"
   if {$calx != ""} then {:spectrum $gallery spectrum calibrate $calx}

   :spectrum $gallery cut y $y $matrix $cuty
   :spectrum $gallery spectrum current $cuty
   :spectrum $gallery spectrum title "$titley (unsaved)"
   if {$caly != ""} then {:spectrum $gallery spectrum calibrate $caly}

   :spectrum $gallery configure -select all

   set ${cf}(cutptrs) [list $cutx $cuty]
   set ${cf}(cuts) "0 1"

   clear-footer
   change-frame $frameSAVE
}

#   called from poly procedures

proc poly-project-current {vs} {
   global frame frames frametitle gallery
   global cuts_height
   global debug

   if {[select spectrum] == ""} then {
      set-footer "no spectrum selected!!!"
      clear-tags
      return
   }

   if {[midas-check-for-temp] == 0} then {return}

   set info [:spectrum $gallery spectrum info]

   set s [transparent [select spectrum]]
# is spectrum a SOAP path
   set sv [inform ex obtain_SOAPService_byName $s]
   if {$sv != ""} {
# build spectrum name in current (temp) directory, replacing /'s in path with _ (e.g. /MemSas/Spect => /tmp/tcl123/MemSAS_Spect)
     set ns [join [tail [split $s /]] _]
     set ts [file join [pwd] $ns]
     catch {eg "(spectrum :path '$ts' !delete)"}
     catch {eval exec copy-soap-spectrum $s $ts} m
   } else {
     set ts $s
   }

   set z [catch {eval exec poly-project \
          -spectrum $ts \
          -poly [flatten $vs]} ps]

   if {$debug} then {puts stdout "poly-project $vs => $z $ps"}

   if {$sv != ""} {
     catch {eg "(spectrum :path '$ts' !delete)"}
   }

   if {$z != 0} then {
      set-footer $ps
      return
    }

   set cutx "[select spectrum]@x"
   set cuty "[select spectrum]@y"
# The following 2 lines ARE correct
# the cut@x is onto the y-axis so it needs the y calibration & vice versa
   set calx [:spectrum $gallery spectrum calibrate y]
   set caly [:spectrum $gallery spectrum calibrate x]

#    now proceed to display the result of the projection

   set frameSAVE $frame

   global $frame
   set cf [set ${frame}(cutframe)]
   if {$cf == "" || [lsearch $frames $cf] < 0} then {
      set cf [new-sd-frame]
      global $cf
      set ${frame}(cutframe) $cf
      set ${cf}(cutframe) $frame
      set ${cf}(framemode) "cut"
      change-frame $cf
      set frametitle "Displaying 2d Cuts/Projections ($frame)"
      init-sd-frame  $cuts_height
   } else {
      global $cf
      change-frame $cf
   }

   set ${cf}(cutptrs) {}
   :spectrum $gallery spectrum clear

   :spectrum $gallery spectrum add-dataset cutx [third $info] [fourth $info]
   :spectrum $gallery spectrum data [third $info] [fourth $info] [second [second $ps]]
   :spectrum $gallery spectrum title "$cutx (unsaved)"
   if {$calx != ""} then {:spectrum $gallery spectrum calibrate $calx}

   :spectrum $gallery spectrum add-dataset cuty [first $info] [second $info]
   :spectrum $gallery spectrum data [first $info] [second $info] [second [first $ps]]
   :spectrum $gallery spectrum title "$cuty (unsaved)"
   if {$caly != ""} then {:spectrum $gallery spectrum calibrate $caly}

   set ${cf}(cuts) "0 1"

   clear-footer
   change-frame $frameSAVE
}


#    some miscellaneous procedures

proc flatten ps {
   set ll {}
   foreach p $ps {lappend ll [first $p] [second $p]}
   return $ll
}

proc member {n ns} {return [expr [lsearch -exact $ns $n]>=0]}


proc select what {
   global frame gallery viewmode
   switch $what {
       frame {
          if {$frame != {}} {return $frame}
          error "no spectrum display frame selected!!!"
       }
       box   {
          set b [:spectrum $gallery get box]
          if {$b != {}} {return $b}
          error "no spectrum display box selected!!!"
       }
       box-num  {
          set b [select box]
          return 0
       }
       spectrum {
          set s [:spectrum $gallery spectrum name]
          set tmp0 [split $s :]
          if {[first $tmp0] == "EGlib"} then {
            set s [join [tail $tmp0] :]
          } else {
            set tmp1 [split $s /]
            if {[llength [first $tmp1]] == 0} then {
               if {[inform ex obtain_SOAPService_byName $s] == {}} {
                  error "Invalid spectrum name $s"
               }
            } else {
               if {[llength $tmp1] != 1} then {
                  set svr_port [first $tmp1]
                  set urn [second $tmp1]
                  set name [inform ex obtain_SOAPServiceName $svr_port $urn]
                  set s [join [list {} $name [tail [tail $tmp1]]] /]
               }
            }
          }
          return $s

          if {$s != {}} {return $s}
          error "no spectrum currently selected!!!"
       }
       view  {
          if {"$viewmode" == "all"} {return "-1"}
          return [select box]
       }
       default {}
   }
}

#    procedures called by external tasks (cal, cut, opt, pdb)
#     (other tasks call proc show)

proc centre-at e {
   global gallery
   if {[set s [select spectrum]] == ""} then {set-footer "no spectrum selected!!!"; return}
   if {[midas-check-for-temp] == 0} then {return}
# is spectrum a SOAP path
   set sv [inform ex obtain_SOAPService_byName $s]
   if {$sv != ""} {
# build spectrum name in current (temp) directory, replacing /'s in path with _ (e.g. /MemSas/Spect => /tmp/tcl123/MemSAS_Spect)
     set ns [join [tail [split $s /]] _]
     set ts [file join [pwd] $ns]
     catch {eg "(spectrum :path '$ts' !delete)"}
     catch {eval exec copy-soap-spectrum $s $ts} m
   } else {
     set ts $s
   }

   set z [catch {exec peakshow -spectrum $ts -peak $e} m]
   if {$sv != ""} {
     catch {eg "(spectrum :path '$ts' !delete)"}
   }
   if {$z != 0} then {set-footer "$m"; return}
   set lastchannel [expr $m-50]
   set currentchannel [expr $m+50]
   :spectrum $gallery view set $lastchannel $currentchannel
}

proc get-tag-and-pop {} {
   global frame gallery
   set tags [get-tags 1]
   if {[llength $tags] < 1} then {return}
   set c [first $tags]
   :spectrum $gallery tag pop 1
   return $c
}

proc get-all-xtags {} {
   global frame gallery debug
   switch [:spectrum $gallery get dimension] {
      "1" {
          set tags [:spectrum $gallery tag get 99]
      }
      "2" {
          set t [:spectrum $gallery tag get2d 99]
          set l [llength $t]
          set tags {}
          for {set i 0} {$i < $l} {incr i 2} {lappend tags [lindex $t $i]}
      }
      default {set tags {}}
   }
   if {$debug} then {puts stdout "got-all-xtags: $tags"}
   return $tags
}

proc get-all-ytags {} {
   global frame gallery debug
    set tags {}
    if {[:spectrum $gallery get dimension] == "2"} {
       set t [:spectrum $gallery tag get2d 99]
       set l [llength $t]
       for {set i 1} {$i < $l} {incr i 2} {lappend tags [lindex $t $i]}
    }
   if {$debug} then {puts stdout "got-all-ytags: $tags"}
   return $tags
}


loadsource sd.analysis
loadsource sd.poly

   set lib spectrum
   set library [set lib]_[join [split [info tclversion] .] ""][info sharedlibextension]
   set z [catch {load $library spectra}]

   if !{$z} {
      if {[info commands __spectrum__] == "" && [info commands spectrum] == "spectrum"} then {rename spectrum __spectrum__} ;# rename original spectrum library command once 
      proc spectrum args {
         global spectrumTrace
         if {$spectrumTrace} then {insert-debug "spectrum calling spectra $args"}
         spectra $args
      }
   } else {
      insert-debug "Couldn't load spectrum display module for Tcl/Tk [info tclversion]"
   }

   set spectrumTrace 0


:frame .sd
