# ex.tcl---experiment access

#  17th Aug 1995  VP  TK migration
# Tue Jan 31 18:51:16 GMT 1995, DB
#    added save-as-default button
# Fri Mar  5 15:28:26 GMT 1993, DB
#    added auto-join proc---joins experiment if EXPERIMENTDIR set
# 29-10-92 PHO
#    added apparatus access button
# Wed Oct 28 15:46:35 GMT 1992, DB
#    changed connect/disconnect nomenclature to join/leave
# Thu Oct 22 17:16:44 BST 1992, DB
#    added directoryname globbing and more informative footers
#    removed password: field
# 20 Aug 92 PHO
#    implemented help widget to call manual browser

set debug 0
set started 0

set frame EX
set frametitle "Accessing an Experiment"
set icontitle  "ExptAccess"

set directory "$env(MIDASEXPBASE)"

#    "capabilities" files cannot be shared between processors having different "endianness"
if {$env(EXPUSEENDIAN) == "true"} then {set endian $env(ENDIAN)} else {set endian "big"}

set filename "capabilities"

set experiment ""
set currentexperiment ""
set joined 0

set resources {}

set hdrcolour yellow

set apparatusnamechoice [list vxi vme camac tapeserver hist VMEhist MemSas SortSas sigtask netvar]
set apparatushostchoice [list "localhost"]
set apparatusrolechoice [list VXI VME CAMAC camac tape hist sort Spectrum Hist SigTask NetVar]
set apparatustypechoice [list register spectrum tape SOAP/XML]
set apparatusportchoice [list 10220 10230 10205 8015]

set apparatus_name [first $apparatusnamechoice]
set apparatus_host [first $apparatushostchoice]
set apparatus_role [first $apparatusrolechoice]
set apparatus_type [first $apparatustypechoice]
set apparatus_port [first $apparatusportchoice]

set apparatus_ports(register) 10220
set apparatus_ports(spectrum) 10230
set apparatus_ports(tape) 10205
set apparatus_ports(SOAP/XML) 8015

proc dirnames l {
   set ll {}
   foreach f $l {lappend ll [first $f]}
   return $ll
}

proc format-resource ps {
   foreach p $ps {
      switch [first $p]  {
         name       {set name [second $p]}
         host       {set host [second $p]}
         port       {set port [second $p]}
         type       {set type [second $p]}
         role       {set role [second $p]}
         apparatus  {set app [second $p]}
         ipaddress  {set ipaddress [second $p]}
         capability {set cap [second $p]}
         flags      {set flags [second $p]}
         default    {}
      }
   }
   set hostport "$host/$port"
   format "%-12s%-10s%-10s%-16s%-10s%-4s%-17s" $name $role $type $hostport \
      $app $flags $cap
}

proc make-resource-list rs {
   set ll {}
   foreach r $rs {lappend ll [format-resource $r]}
   return $ll
}

proc read-resources {} {
   global resources
   set resources [:experiment resources]
}

proc click-join {} {
   global directory filename experiment currentexperiment joined env endian
   clear-footer
   if {$experiment==""} then {
      set-footer "You haven't selected an experiment to join!!!"
      return
   }
   if {![file exists [file join $directory $experiment]]} then  \
     {midas-warning "Required experiment file does not exist - use Create first\n\
                    Experiment NOT joined"; return}
   if {$endian == "big"} then {
      if {![file exists [file join $directory $experiment $filename]]} then  \
         {midas-warning "Required experiment file does not exist for this processor type- use Create first\n\
                        Experiment NOT joined"; return}
      if {![file writable [file join $directory $experiment $filename]]} then  \
         {midas-warning "You do not have write permission for this experiment file\n\
                        Experiment NOT joined"; return}
      set-footer "Joining experiment $experiment ..."
      :experiment connect :path [file join $directory $experiment] :name $filename
      set env(EXPERIMENTDIR) [file join $directory $experiment]
   } else {
      if {![file exists [file join $directory $experiment $endian $filename]]} then  \
         {midas-warning "Required experiment file does not exist for this processor type- use Create first\n\
                        Experiment NOT joined"; return}
      if {![file writable [file join $directory $experiment $endian $filename]]} then  \
         {midas-warning "You do not have write permission for this experiment file\n\
                        Experiment NOT joined"; return}
      set-footer "Joining experiment $experiment ..."
      :experiment connect :path [file join $directory $experiment $endian] :name $filename
      set env(EXPERIMENTDIR) [file join $directory $experiment $endian]
   }
   set env(EXPERIMENTBASE) [file join $directory $experiment]
   set currentexperiment $experiment
   read_SOAPServers                      ;#   ignore all errors including no servers of this type
   read_RPCServers
   set joined 1
   broadcast-experiment
   repaint
   set-footer "You are now working with experiment $experiment"
   insert-log "You are now working with experiment $experiment"
}

proc auto-join {} {
   global env started joined
   global directory experiment

   if {$started == 0} then {make-frame; close-frame}

   set directory  [file dirname $env(EXPERIMENTBASE)]
   set experiment [file tail $env(EXPERIMENTBASE)]
   click-join
   if {$joined == 0} {unset env(EXPERIMENTBASE)}
}

proc broadcast-experiment {} {
   global directory experiment
#     send experiment set-experiment $directory/$experiment

   foreach task [Get_Task_Names] {
      if {[inform $task "info procs set-experiment"] == "set-experiment"} then {
          inform $task set-experiment [file join $directory $experiment]
      }
   }
}

proc click-leave {} {
   global directory experiment currentexperiment joined env
   set-footer "Leaving experiment ..."
   :experiment disconnect
   set currentexperiment ""
   unread_SOAPServers
   unread_RPCServers
   unset env(EXPERIMENTDIR)
   unset env(EXPERIMENTBASE)
   set joined 0
   unbroadcast-experiment
   repaint
   set-footer "You are no longer working with any experiment"
   insert-log "You are no longer working with any experiment"
}

proc unbroadcast-experiment {} {
#     send experiment unset-experiment 

   foreach task [Get_Task_Names] {
      if {[inform $task "info procs unset-experiment"] == "unset-experiment"} then {
          inform $task unset-experiment 
      }
   }
}

proc click-create {} {
   global endian directory filename experiment currentexperiment joined
   clear-footer
   if {$experiment==""} then {
      midas-warning "Failed to create experiment\n\nPlease supply a name for the new experiment"
      return
   }
   set-footer "Creating experiment $experiment ..."
   set z [catch {:file mkdir [file join $directory $experiment]} m]
   if {$z!=0} then {
      midas-error "" "Experiment $experiment not created" $m
      return
   }
   if {$endian != "big"} then {
      set z [catch {:file mkdir [file join $directory $experiment $endian]} m]
      if {$z!=0} then {
         midas-error "" "Experiment $experiment not created" $m
         return
      }
   }
   if {$endian == "big"} then {
      :experiment new :path [file join $directory $experiment] :name $filename
   } else {
      :experiment new :path [file join $directory $experiment $endian] :name $filename
   }
   paint-directory
   set-footer "Experiment $experiment successfully created"
}

proc click-delete {} {
   global directory filename experiment currentexperiment joined
   clear-footer
   if {$experiment==""} then {
      midas-warning "Failed to delete experiment\n\nPlease select the experiment to be deleted"
      return
   }
   set-footer "Deleting experiment $experiment ..."
   :experiment delete :path [file join $directory $experiment] :name $filename
   set z [catch {:file delete -r [file join $directory $experiment]} m]
   if {$z!=0} then {
      midas-error "" "Experiment $experiment not deleted" $m
      return
   }
   paint-directory
   set-footer "Experiment $experiment now deleted"
}

proc edit-defaults {key stuff} {
   global debug
   if $debug then {puts stdout $key; puts stdout $stuff}
   set lines [load-defaults]
   set range [find-keyword $lines $key]
   if {[llength $range] == 0} then {
       save-defaults [linsert $lines [llength $lines] "## $key\n$stuff"]
   } else {
       save-defaults [lreplace $lines [first $range] [second $range] "## $key\n$stuff"]
   }
}

proc load-defaults {} {
   global env
   set z [catch {set f [open [file join $env(HOME) .eg-defaults.tcl] r]}]
   if $z!=0 then {return [list "# eg-session defaults for user $env(USER)" ""]}
   set lines [split [read $f] "\n"]
   close $f
   return $lines
}

proc save-defaults {lines} {
   global env
   set f [open [file join $env(HOME) .eg-defaults.tcl] w]
   puts $f [join $lines "\n"]
   close $f
}

proc find-keyword {lines key} {
   set i 0; set found 0
   foreach l $lines {
      switch $found  {
         0   {if {[string first "## $key" $l]==0} then {set start $i; set found 1}}
         1   {if {[string first "##" $l]==0} then {set finish $i; set found 2}}
         2   {break}
      }
     incr i
   }
   if $found==0 then {return {}}
   if $found==1 then {return [list $start [expr $i-1]]}
   if $found==2 then {return [list $start [expr $finish-1]]}
}

proc click-save {} {
   global debug joined env
   if {$joined} then {
      edit-defaults experiment-directory "set-eg-option experiment-directory $env(EXPERIMENTBASE)"
   } else {
      edit-defaults experiment-directory ""
   }
}

proc click-inspect {} {
   clear-footer
   make-framer
   read-resources
   paint-resources
}

proc click-directory d {
   global directory
   clear-footer
   set directory [glob $d]
   paint-directory
}

proc click-experiment e {
   global experiment
   clear-footer
   set experiment $e
}

proc repaint {} {
   global env directory experiment currentexperiment

    :modify DIRECTORY  :value "$directory"
    :modify EXPERIMENT :value "$experiment"
    :modify CURRENT    :value "$currentexperiment"
}

proc paint-resources {} {
   global env resources currentexperiment
   global SOAPServer

    :modify [self frame-name].insp :label "Resource Inspection for: $currentexperiment"

    :modify EXR-LIST :empty T 

    set RPCresources [make-resource-list $resources]
    if {[llength $RPCresources] == 0} then {
        :modify EXR-LIST :usingtag taghdr :strings "There are no RPC resources\n"
    } else {
        :modify EXR-LIST :usingtag taghdr :strings "RPC resources\n"
        :modify EXR-LIST :usingtag taghdr :strings " name        role       type      host/port   apparatus  status   capability\n"
        foreach line [make-resource-list $resources] {:modify EXR-LIST  :strings "$line\n"}
    }

    read_SOAPServers
 
    set SOAPresources [array names SOAPServer]

    if {[llength $SOAPresources] <= 1} then {
        :modify EXR-LIST :usingtag taghdr :strings "\nThere are no SOAP/XML resources\n"
    } else {
        :modify EXR-LIST :usingtag taghdr :strings "\nSOAP resources\n"
        :modify EXR-LIST :usingtag taghdr :strings " name        location       role         type\n"
        foreach  item  $SOAPresources {
            if {$item == "Dummy"} {continue}
            :modify EXR-LIST  :strings "[format "%-12s%-16s%-10s%-10s" $item [first $SOAPServer($item)] [second $SOAPServer($item)] [third $SOAPServer($item)]]\n"
        }
    }

    :modify EXR-LIST :strings "" :see 0
}

proc paint-directory {} {
   global env
   global directory
   switch $env(platform) {
      windows {set dirlist [lsort [strip-dots [dirnames [:directory names :path $directory]]]]}
      default {set dirlist [lsort [strip-dots [dirnames [:directory files :path $directory]]]]}
   }
   :modify LIST :empty T :append T :strings $dirlist :see 0
}

proc click-list {v} {
   global experiment
   clear-footer
   set experiment $v
   repaint
}

proc double-click arg {
   global debug

   if {$debug} then {puts stderr "double-click with argument \{$arg\}"}
# insert-debug "double-click entered with argument \{$arg\}"
   set f [lindex $arg 0]
   set w [lindex $arg 2]
   set v [lindex $arg 3]

   switch -glob $w {
        *LIST  {
          global EGallow
          click-list $v
          set ega $EGallow; set EGallow 0x10008
          catch {click-leave}; click-join
          set EGallow  $ega
        }
        default      {puts stdout $arg}
   }
}

proc click-quit f {
   global debug started
   if {$debug} then {puts stderr "click-quit with argument \{$f\}" }
   destroy-frame $f
   set started 0
}

proc start args {
   global started env joined currentexperiment experiment

   if {$started == 0} then {make-frame} else {open-frame}
   set started 1
   if {[info exists env(EXPERIMENTBASE)]} then {
       set experiment [file tail $env(EXPERIMENTBASE)]
       set currentexperiment $experiment
       read_RPCServers                      ;#   ignore all errors including no servers of this type
       read_SOAPServers                      ;#   ignore all errors including no servers of this type
       set joined 1
   }
   paint-directory
   repaint
}

proc finish args {destroy-frame}


proc make-frame {} {
  global directory

     :frame [self frame-name] :label "[self frame-title]" :show-footer T \
       :icon-name "[self icon-title]" :resize F

     :panel INFO :layout L :padx 5 :pady 5
        :layout V
        :row-gap 5
        :text DIRECTORY  :label "Directory:"          :w 20  :valuex 20c :value "$directory"
        :text EXPERIMENT :label "Experiment:"         :w 16  :valuex 20c :value ""
        :text CURRENT    :label "Current Experiment:" :w 16  :valuex 20c :value "" :deaf T

     :panel L :layout L :pady 5
        :list LIST :rows 7 :cols 18  :double-click T

     :panel B1 :layout B :padx 5 :after FOOTER
       :layout H; :col-gap 10
        :button JOIN
        :button LEAVE
        :button CREATE
        :button DELETE
        :button SAVE   :label "Save as default"

       :next-row T
        :button INSPECT     :label "Inspect Resources"
        :button APPARATUS   :label "Apparatus Control"
        :button HELP

     :show T
}

proc make-framer {} {
  global hdrcolour
   :frame [self frame-name].insp :transient T
   :layout B
   :galley EXR-LIST  :rows 12 :cols 86 :deaf T :font fixed
   :modify EXR-LIST  :createtag "taghdr -foreground $hdrcolour"
   :show T
}

proc make-apparatusframe {} {
   global apparatusnamechoice apparatushostchoice apparatusrolechoice apparatustypechoice apparatusportchoice
   global apparatus_name apparatus_host apparatus_role apparatus_type apparatus_port

set w 20

   :frame [self frame-name].apparatus :label "Experiment Apparatus Control" :show-footer T \
       :icon-name "[self icon-title]" :resize F

   :panel APP :layout L :padx 5 :pady 5
   :layout H; :row-gap 10

   :text APPNAME            :label "Apparatus Name"  :valuex 16c  :w $w  :value $apparatus_name \
         :command apparatus_input
   :menubutton APPNAMEMENU  :label "Select"  :strings $apparatusnamechoice   :command apparatus_input
   :next-row T

   :text APPHOST            :label "Apparatus Host"  :valuex 16c  :w $w  :value $apparatus_host \
         :command apparatus_input
   :menubutton APPHOSTMENU  :label "Select"  :strings $apparatushostchoice   :command apparatus_input
   :next-row T

   :text APPROLE            :label "Apparatus Role"  :valuex 16c  :w $w  :value $apparatus_role \
         :command apparatus_input
   :menubutton APPROLEMENU  :label "Select"  :strings $apparatusrolechoice   :command apparatus_input
   :next-row T

   :text APPTYPE            :label "Apparatus Type"  :valuex 16c  :w $w  :value $apparatus_type \
         :command apparatus_input
   :menubutton APPTYPEMENU  :label "Select"  :strings $apparatustypechoice   :command apparatus_input
   :next-row T

   :text APPPORT            :label "Apparatus Port"  :valuex 16c  :w $w  :value $apparatus_port \
         :command apparatus_input
   :menubutton APPPORTMENU  :label "Select"  :strings $apparatusportchoice   :command apparatus_input
   :next-row T

   :col-gap 10
   :button APPADD  :label "Add apparatus"      :command add_apparatus
   :button APPREM  :label "Remove apparatus"   :command rem_apparatus

   :show T
}

proc click-apparatus {} {make-apparatusframe}

proc apparatus_input {w vl args} {
    global apparatus_access apparatus_name apparatus_host apparatus_role apparatus_type apparatus_port
    global apparatusportchoice

    switch $w {
        APPNAMEMENU  -
        APPNAME  {set apparatus_name $vl; :modify APPNAME :value $vl}
        APPHOSTMENU  -
        APPHOST  {set apparatus_host $vl; :modify APPHOST :value $vl}
        APPROLEMENU  -
        APPROLE  {set apparatus_role $vl; :modify APPROLE :value $vl}
        APPTYPEMENU  -
        APPTYPE  {set apparatus_type $vl
                  :modify APPTYPE :value $vl
                  set apparatus_port [lindex $apparatusportchoice [first $args]]
                  :modify APPPORT :value $apparatus_port
            }
        APPPORTMENU  -
        APPPORT  {set apparatus_port $vl; :modify APPPORT :value $vl}
   }
}

proc add_apparatus {} {
    global apparatus_type

   if {$apparatus_type == "SOAP/XML"} then {add_SOAP_apparatus} else {add_RPC_apparatus}
}

proc rem_apparatus {} {
    global apparatus_type

   if {$apparatus_type == "SOAP/XML"} then {rem_SOAP_apparatus} else {rem_RPC_apparatus}
}

proc add_RPC_apparatus {} {
    global env apparatus_name apparatus_host apparatus_role apparatus_type apparatus_port
    
#    check for and is necessary create a temporary directory
    if {[midas-check-for-temp] == 0} {return}

    set directory [file join $env(MIDASTempDir) tcl[pid]]
    set file $apparatus_name

#    create apparatus file
    set z [catch {open [file join $directory $file] w} f]
    if {$z != 0} then {midas-warning "Unable to write [file join $directory $file]\n$f"; return}

    puts $f "((name $apparatus_name)"
    puts $f " (host $apparatus_host)"
    puts $f " (role $apparatus_role)"
    puts $f " (type $apparatus_type)"
    puts $f " (ipaddress 123456)"
    puts $f " (port $apparatus_port))"

    close $f

#    attach apparatus
#   :apparatus attach :path $directory :name $file

    set path $directory
    set name $file

          set z [catch {eg "(apparatus :path '$path' :name '$name' !attach)"} m]
          if {$z != 0} then {
             display-midas-error-report self $m "Attempting to attach apparatus file" $name
             set z [catch {eg "(apparatus :name '$name' !detach)"} m]
             return
          }

#    and claim it
#   :apparatus claim :name $file

          set z [catch {eg "(apparatus :name '$name' !claim)"} m]
          if {$z != 0} then {
             display-midas-error-report self $m "Attempting to claim apparatus" $name
             set z [catch {eg "(apparatus :name '$name' !free)"} m]
             set z [catch {eg "(apparatus :name '$name' !detach)"} m]
             return
          }

    read_RPCServers
}

proc rem_RPC_apparatus {} {
    global apparatus_name

    set file $apparatus_name

#   free it
#    :apparatus free :name $file

    set name $file

          set z [catch {eg "(apparatus :name '$name' !free)"} m]
          if {$z != 0} then {
             display-midas-error-report self $m "Attempting to free apparatus" $name
             set z [catch {eg "(apparatus :name '$name' !detach)"} m]
             return
          }


#    and detach
#    :apparatus detach :name $file

          set z [catch {eg "(apparatus :name '$name' !detach)"} m]
          if {$z != 0} then {
             display-midas-error-report self $m "Attempting to detach apparatus file" $name
             return
          }

    read_RPCServers
}

proc add_SOAP_apparatus {} {
    global env apparatus_name apparatus_host apparatus_role apparatus_type apparatus_port
    global SOAPServer
    
    read_SOAPServers

    switch $apparatus_role {
        CAMAC  -
        VME    {
             set SOAPServer($apparatus_name) "$apparatus_host:$apparatus_port $apparatus_role [set apparatus_role]AccessServer [set apparatus_role]AccessServer [set apparatus_role]AccessClient"
         }
        Spectrum -
        SigTask -
        NetVar    {
             set SOAPServer($apparatus_name) "$apparatus_host:$apparatus_port $apparatus_role [set apparatus_role]Service [set apparatus_role]Service [set apparatus_role]Client"
         }
        Hist    {
             set SOAPServer($apparatus_name) "$apparatus_host:$apparatus_port $apparatus_role [set apparatus_role]Service SpectrumService SpectrumClient"
         }
        tape -
        Tape    {
             set SOAPServer($apparatus_name) "$apparatus_host:$apparatus_port Tape TapeServer TapeServer TapeClient"
         }
        default {
             set SOAPServer($apparatus_name) "$apparatus_host:$apparatus_port $apparatus_role [set apparatus_role]Service [set apparatus_role]Service [set apparatus_role]Client"
        }
    }

    write_SOAPServers
}

proc rem_SOAP_apparatus {} {
    global apparatus_name
    global SOAPServer

    read_SOAPServers

    unset SOAPServer($apparatus_name)

    write_SOAPServers
}

proc read_RPCServers {} {
    global env RPCServer

    catch {unset RPCServer}
    set RPCServer(Dummy) "Dummy"

    set info [:experiment resources]
    foreach item $info {
       if {[llength $item] == 8} {
           foreach element $item {
               switch [first $element] {
                   name  {set name [second $element]}
                   host  {set host [second $element]}
                   port  {set port [second $element]}
                   role  {set role [second $element]}
                   type  {set type [second $element]}
                   capability  {set capability [second $element]}
                   default {}
               }
           }
           if {[expr $capability] != 0} {set RPCServer($name) "[set host]:[set port] $role $type"}
       }
    }
}

proc read_SOAPServers {} {
    global env SOAPServer

    catch {unset SOAPServer}
    set SOAPServer(Dummy) "Dummy"

    set directory $env(EXPERIMENTBASE)
    set file SOAPServers

#    attempt to read SOAP server file
    set z [catch {open [file join $directory $file] r} f]
    if {$z != 0} then {return}
    set info [split [read $f] \n]
    close $f

    foreach item $info {
       if {[llength $item] == 6} {
           set SOAPServer([first $item]) "[second $item] [third $item] [fourth $item] [fifth $item] [sixth $item]"
       }
    }
}

proc write_SOAPServers {} {
    global env SOAPServer
    global apparatus_name apparatus_host apparatus_role apparatus_type apparatus_port

    set directory $env(EXPERIMENTBASE)
    set file SOAPServers

#    attempt to write SOAP server file
    set z [catch {open [file join $directory $file] w} f]
    if {$z != 0} then {return}

    foreach  item  [array names SOAPServer] {
         if {$item == "Dummy"} {continue}
         puts $f "$item $SOAPServer($item)"
    }

    close $f
}

proc unread_RPCServers {} {
    global RPCServer

    catch {unset RPCServer}
}

proc unread_SOAPServers {} {
    global SOAPServer

    catch {unset SOAPServer}
}


proc obtain_Server_byRole {Role} {

    set reply {}

    lappend reply [obtain_RPCServer_byRole $Role]
    lappend reply [obtain_SOAPServer_byRole $Role]

    return $reply
}

proc obtain_SOAPServer_byRole {Role} {
    global SOAPServer

    set reply {}

    foreach  item  [array names SOAPServer] {
         if {$item == "Dummy"} {continue}
         if {[second $SOAPServer($item)] == $Role} then {lappend reply [first $SOAPServer($item)]}
    }

   return $reply
}

proc obtain_RPCServer_byRole {Role} {
    global RPCServer

    set reply {}

    foreach  item  [array names RPCServer] {
         if {$item == "Dummy"} {continue}
         if {[second $RPCServer($item)] == $Role} then {lappend reply [first $RPCServer($item)]}
#         if {[second $RPCServer($item)] == $Role} then {lappend reply $item}
    }

   return $reply
}

proc obtain_SOAPServers_byRole {Role} {return [obtain_SOAPServer_byRole $Role]}

proc obtain_Server_byName {Name} {

    set reply {}

    lappend reply [obtain_RPCServer_byName $Name]
    lappend reply [obtain_SOAPServer_byName $Name]

    return $reply
}

proc obtain_SOAPServer_byName {Name} {
    global SOAPServer

    set reply {}

    foreach  item  [array names SOAPServer] {
         if {$item == "Dummy"} {continue}
         if {$item == $Name} then {lappend reply [first $SOAPServer($item)]}
    }

   return $reply
}

proc obtain_RPCServer_byName {Name} {
    global RPCServer

    set reply {}

    foreach  item  [array names RPCServer] {
         if {$item == "Dummy"} {continue}
         if {$item == $Name} then {lappend reply [first $RPCServer($item)]}
#         if {$item == $Name} then {lappend reply $item}
    }

   return $reply
}

proc obtain_SOAPServers_byName {Name} {
    global SOAPServer

    set reply {}

    foreach  item  [array names SOAPServer] {
         if {$item == "Dummy"} {continue}
         if {[regexp $Name $item]} then {lappend reply [first $SOAPServer($item)]}
    }

   return $reply
}

proc obtain_RPCServerInfo {Server Name} {
    global RPCServer

    foreach  item  [array names RPCServer] {   ;# try as a Role
         if {$item == "Dummy"} {continue}
         if {[second $RPCServer($item)] == $Name && [first $RPCServer($item)] == $Server} then {return "$item $RPCServer($item)"}
    }

    foreach  item  [array names RPCServer] {
         if {$item == "Dummy"} {continue}
         if {$item == $Name && [first $RPCServer($item)] == $Server} then {return "$item $RPCServer($item)"}
    }

   return ""
}

proc obtain_SOAPServerInfo {Server Name} {
    global SOAPServer

    foreach  item  [array names SOAPServer] {  ;# try as a Role
         if {$item == "Dummy"} {continue}
         if {[second $SOAPServer($item)] == $Name && [first $SOAPServer($item)] == $Server} then {return "$item $SOAPServer($item)"}
    }

    foreach  item  [array names SOAPServer] {
         if {$item == "Dummy"} {continue}
         if {$item == $Name && [first $SOAPServer($item)] == $Server} then {return "$item $SOAPServer($item)"}
    }

   return ""
}

proc obtain_SOAPService_byName {Name} {
    global SOAPServer

    set reply {}
    set nm [split $Name /]
    if {[set Name [first $nm]] == {}} then {
      set Name [second $nm]
    }
    foreach  item  [array names SOAPServer] {
         if {$item != $Name} {continue}
         lappend reply [first $SOAPServer($item)]/[fourth $SOAPServer($item)]
    }

   return $reply
}

proc obtain_SOAPServiceName {Server Urn} {
    global SOAPServer

    foreach  item  [array names SOAPServer] {
         if {$item == "Dummy"} {continue}
         if {[first $SOAPServer($item)] == $Server && [fourth $SOAPServer($item)] == $Urn} then {return "$item"}
    }

   return ""
}
