# Spectrum Calibration application

# Wed Aug 10 14:34:18 BST 1994, DB (copied version)

package require ServerAccess 2.0
package require SpectrumClient

set started 0

set frame CAL
set frametitle "Calibrating Spectra"
set icontitle Calibrating

set axis X
set axismenu [list X Y Z]
set order linear
set ordermenu  [list linear quadratic cubic]
set units KeV
set unitsmenu  [list KeV MeV ns us ms]

set chan ""
set chanerr ""
set calib ""
set caliberr ""

set pointlist {{238 .3 121.78 0.044}
               {1548 .7 778.9 0.08}
               {2800 .11 1408.01 0.012}}

set pointstrings {}
set selection -1
set spectrum ""
set latestfit {}

set browserdirectory  $env(HOME)
set browserfile ""

proc do-saveinformation {args} {
   upvar #0 browserdirectory directory
   upvar #0 browserfile file
   upvar #0 pointlist information

   set z [catch {open $directory/$file w} f]
   if {$z != 0} then {midas-warning "Unable to write $directory/$file\n$f"; return}
   foreach item $information {puts $f $item}
   close $f
}

proc do-readinformation {args} {
   upvar #0 browserdirectory directory
   upvar #0 browserfile file
   upvar #0 pointlist information

   set z [catch {open $directory/$file r} f]
   if {$z != 0} then {midas-warning "Unable to read $directory/$file\n$f"; return}
   set information [split [read $f] \n]
   close $f
}

proc click-save {} {midas-browser W do-saveinformation}
proc click-load {} {midas-browser R do-readinformation; click-redisplay}

proc announce-fit {report} {
   global latestfit
   set latestfit $report
}

proc repaint {} {
   global axismenu axis ordermenu order unitsmenu units chan chanerr calib caliberr spectrum 
   :modify AXIS :value [lsearch $axismenu $axis]
   :modify ORDER :value [lsearch $ordermenu $order]
   :modify UNITS :value [lsearch $unitsmenu $units]
   :modify CHAN :value "$chan"
   :modify CHAN-ERR :value "$chanerr"
   :modify CALIB :value "$calib"
   :modify CALIB-ERR :value "$caliberr"
   :modify SPECTRUM :value $spectrum
}

proc format-points l {
   set ll {}
   foreach f $l {
      if {"$f" == ""} {continue}
      lappend ll [format "%8.3f (%4.3f)      %8.3f (%4.3f)" [first $f] [second $f] [third $f] [fourth $f]]
   }
   return $ll
}

proc clear-point {} {
   global chan chanerr calib caliberr
   set chan ""; set chanerr ""; set calib ""; set caliberr ""
}

proc load-point n {
   global chan chanerr calib caliberr pointlist
   set point [lindex $pointlist $n]
   set chan [first $point]
   set chanerr [second $point]
   set calib [third $point]
   set caliberr [fourth $point]
}

proc paint-pointlist {} {
   global pointlist pointstrings selection
   set pointstrings [format-points $pointlist]
   :modify POINTLIST :empty T :strings $pointstrings
   :modify POINTLIST  :select $selection
}

proc list-take {as is} {
   set ll {}
   foreach e $is {lappend ll [lindex $as $e]}
   return $ll
}

proc click-takefit {} {
   global latestfit chan chanerr spectrum
   if {$latestfit == {}} then {
      set-footer "you haven't performed a peak fit!!!"
      return
   }
   set fitlines [split $latestfit "\n"]
   if {[llength $fitlines] < 3} then {
      set-footer "sorry, can't process this fit---too few lines"
      return
   }
   set titleline [first $fitlines]
   set titlewords [split $titleline " ,"]
   if {[first $titlewords]!="Background/Peak"} then {
      set-footer "sorry, can't recognise this fit---not made by 'backpeak'"
      return
   }
   set s [lindex $titlewords 4]
   if {$s != ""} then {set spectrum $s}
   set peakline [third $fitlines]
   set z [scan $peakline "%f (%f)" chan chanerr]
   if {$z != 2} then {
      set-footer "sorry, can't process this fit---too few values"
      return
   }
   repaint
   clear-footer
}

proc check-point {} {
   global order units chan chanerr calib caliberr pointlist
   if {[scan $chan "%f" chan] !=1 } then {return 1}
   if {[scan $chanerr "%f" chanerr] !=1 } then {return 1}
   if {[scan $calib "%f" calib] !=1 } then {return 1}
   if {[scan $caliberr "%f" caliberr] !=1 } then {return 1}
   return 0
}

proc click-add {} {
   global order units chan chanerr calib caliberr pointlist
   if {[check-point] != 0} then {
      set-footer "you haven't supplied enough data for a calibration point!!!"
      return
   }
   lappend pointlist [list $chan $chanerr $calib $caliberr]
   paint-pointlist
   clear-footer
}

proc click-replace {} {
   global selection order units chan chanerr calib caliberr pointlist
   if {$selection < 0} then {
      set-footer "you haven't selected a point to replace!"
      return
   }
   set pointlist [lreplace $pointlist $selection $selection \
        [list $chan $chanerr $calib $caliberr]]
   paint-pointlist
   clear-footer
}

proc click-delete {} {
   global selection pointlist
   if {$selection < 0} then {
      set-footer "you haven't selected a point to delete!"
      return
   }
   set pointlist [lreplace $pointlist $selection $selection]
   set selection -1
   paint-pointlist
   clear-footer
}

proc click-locate {} {
   global selection pointlist
   if {$selection < 0} then {
      set-footer "you haven't selected a point to locate!"
      return
   }
   inform sd centre-at [third [lindex $pointlist $selection]]
   clear-footer
}

proc make-cal-input {} {
   global pointlist
   set ll {}
   foreach p $pointlist {
      append ll "[first $p]  [third $p] [second $p] [fourth $p] \n"
   }
   return $ll
}

proc click-calibrate ss {
   global spectrum order units axis
   if {[llength $ss] == 0 && $spectrum == ""} then {
      set-footer "no spectrum selected!!!"; clear-tags
      return
   }
   if {[llength $ss] ==0 } then {set ss [list $spectrum]}
   foreach s $ss {
      midas-check-for-temp
      set ns [last [split $s /]]
      set ts [join [list [pwd] $ns] /]
      set sd [inform ex obtain_SOAPService_byName $s]
      if {$sd != {}} {
         set cs [join [list $sd [tail [tail [split $s /]]]] /]
      }
      if {$cs != $s} {
         catch {eg "(spectrum :path '$ts' !delete)"}
         catch {eval exec copy-soap-spectrum $cs $ts} m
      } else {
         set ts $cs
      }
      set z [catch {exec calibrate -spectrum $ts \
         -axis $axis -opt $order -units $units << [make-cal-input] } m]
      if {$z !=  0} then {
         set-footer "[first [split $m \n]]"
      } else {
         set-footer "spectrum $s calibrated"
         if {$cs != $s} {
            catch {eg "(spectrum :path '$s' !delete)"}
            catch {eval exec copy-soap-spectrum $ts $cs} nn
         }
         insert-log $m
      }
      if {$cs != $s} {
         catch {eg "(spectrum :path '$ts' !delete)"}
      }
   }
   inform sd update-calibrations
}

proc click-pointlist {s} {
   global pointstrings selection
   if {$s == $selection} then {
      set selection -1
   }  else {
      set selection $s
      load-point $selection
   }
   :modify POINTLIST :select $selection
   repaint
   clear-footer
}

proc click-text {p v} {
   global debug
   if {$debug} then {puts stdout "$v"}
   global $p
   set z [scan $v "%f" zz]
   if {$z==0} then {set-footer "$v is not an acceptable number"; return}
   set $p $v
   clear-footer
}

proc click-redisplay {} {
   paint-pointlist
   clear-footer
}

proc click arg {
   global debug
   global spectrum axis order units
   if {$debug} then {puts stdout "click with argument \{$arg\}"}
# insert-debug "click $arg"
   set f [lindex $arg 0]
   set c [lindex $arg 1]
   set w [lindex $arg 2]
   set v [lindex $arg 3]
   set i [lindex $arg 4]
   if {$c==":quit"} then {click-quit; return}
   switch $w {
      ADD         {click-add}
      REPLACE     {click-replace}
      DELETE      {click-delete}
      LOCATE      {click-locate}
      TAKEFIT     {click-takefit}
      CALIBRATE   {click-calibrate {}}
      POINTLIST   {click-pointlist $i}
      AXIS        {set axis $i}
      ORDER       {set order $i}
      UNITS       {set units $i}
      CHAN        {click-text chan $v}
      CHAN-ERR    {click-text chanerr $v}
      CALIB       {click-text calib $v}
      CALIB-ERR   {click-text caliberr $v}
      SPECTRUM    {set spectrum $v}
      LOAD        {click-load}
      SAVE        {click-save}
      HELP        {click-help}
      REDISPLAY   {click-redisplay}
      default    {puts stdout $arg}
   }
}

proc start args {
   global started spectrum
   if {[llength $args] == 1} then {set args [first $args]}
   if {$started  == 0} then {make-frame; set started 1} else {open-frame}
   if {[llength $args] > 0} then {set spectrum [first $args]}
   repaint
   click-redisplay
}


proc make-frame {} {
   global env axismenu ordermenu unitsmenu

   :frame [self frame-name] :label "[self frame-title]" :show-footer T  \
      :icon-image "@$env(ICONHOME)/spectrum"  \
      :icon-name [self icon-title]

    :panel P1 :layout L :pady 5 :padx 10
     :layout V; :row-gap 5
     :choice-stack AXIS :label Axis: :exclusive T :valuex 12c :strings $axismenu :value 0
     :choice-stack ORDER :label "Poly Order:" :exclusive T :valuex 12c :strings $ordermenu :value 1
     :choice-stack UNITS :label Units: :exclusive T :valuex 12c :strings $unitsmenu :value 0
     :text CHAN     :label Channel: :w 12 :valuex 12c
     :text CHAN-ERR :label Error:   :w 12 :valuex 12c
     :text CALIB    :label Calibration: :w 12 :valuex 12c
     :text CALIB-ERR :label Error   :w 12 :valuex 12c

    :panel P2 :layout L :pady 5 :padx 5
     :layout V; :row-gap 10
     :message M1 :label "channel (error)   calibration (error)" :bold T
     :list POINTLIST :rows 6 :cols 40 :exclusive T
     :text SPECTRUM :w 40

    :panel P3 :layout B :after FOOTER :pady 5 :padx 10
     :layout H; :row-gap 10; :col-gap 5
     :button ADD; :button REPLACE; :button DELETE; :button LOCATE; :button CALIBRATE
     :button TAKEFIT :label "Take Fit"
     :button LOAD :label "Load"
     :button SAVE :label "Save"
     :next-row T
     :button REDISPLAY
     :hskip 250
     :button HELP

  :show T
}

